//
// 	NetFilterSDK 
// 	Copyright (C) Vitaly Sidorov
//	All rights reserved.
//
//	This file is a part of the NetFilter SDK.
//	The code and information is provided "as-is" without
//	warranty of any kind, either expressed or implied.
//

#include "vdefs.h"
#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>
#include <dlfcn.h>
#include <fcntl.h>
#include <glob.h>
#include <syslog.h>
#include <pwd.h> 
#include <grp.h> 
#include <unistd.h>
#include <signal.h>
#include <stdarg.h>
#include <netdb.h>
#include <limits.h>
#include <string.h>
#include <dirent.h>
#include <ctype.h>
#include <ctime>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <sys/un.h>
#include <sys/errno.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/epoll.h>

#include <net/if.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <ifaddrs.h>
#include <netinet/tcp.h>
#include <sys/sysinfo.h>
#include <sys/resource.h>

#include "nfevents.h"
#include "vbuffer.h"
#include "dbglog.h"
#include "nfthreadpool.h"
#include "hashtable.h"
#include "nfiocp.h"
#include "nfproxy.h"
#include "nfapi_linux.h"
#include "nfrules.h"
#include "nfutil.h"

using namespace mainNS;

#define NF_PACKET_SIZE 8*1024
#define NF_MAX_PACKET_SIZE (128 * 1024)

#define HASHTABLE_SIZE 3001

#define TCP_MAX_QUEUE_SIZE_OUT 3
#define TCP_MAX_QUEUE_SIZE_IN 3

#define INVALID_SOCKET (-1)

#define LISTEN_BACKLOG 50

#define NF_DEFAULT_TIMEOUT 20
#define NF_CONNECT_TIMEOUT 60
#define NF_FIN_WAIT_TIMEOUT 60

enum OV_TYPE
{
	OVT_TCP_ACCEPTED,
	OVT_TCP_CONNECTED,
	OVT_TCP_CLOSED,
	OVT_TCP_LOCAL_EVENT,
	OVT_TCP_REMOTE_EVENT,
	OVT_TCP_LOCAL_WRITE,
	OVT_TCP_LOCAL_READ,
	OVT_TCP_REMOTE_WRITE,
	OVT_TCP_REMOTE_READ,
	OVT_MAX,
};

struct _NFCTX;

typedef struct _OV_DATA
{
	VLIST_ENTRY	entry;
	ENDPOINT_ID id;
	unsigned int type;
	_NFCTX	*	pNfCtx;
	VLIST_ENTRY packetList;
	size_t		transferred;
	int			error;
} OV_DATA, *POV_DATA;

struct PROXY_FLOW
{
	_NFCTX	*	pNfCtx;
	bool		isLocal;
	int 		eventsActive;
	unsigned int bufferSize;
	bool	disconnected;
	bool	receiveInProgress;
	bool	sendInProgress;
	bool	disconnect;
	bool	shut;
	bool	closed;
	VLIST_ENTRY packetList;
	unsigned int packetListSize;
};

enum PROXY_STATE
{
	PS_NONE,
	PS_CONNECTED,
	PS_ERROR,
	PS_CLOSED
};

typedef struct _NFCTX
{
	VLIST_ENTRY		entry;

	ENDPOINT_ID		id;
	PHASH_TABLE_ENTRY id_next;			// Hash linkage

	int				protocol;

	PROXY_FLOW		localFlow;
	PROXY_FLOW		remoteFlow;

	PROXY_STATE		proxyState;
	
	union  
	{
		NF_TCP_CONN_INFO tcpConnInfo;
		NF_UDP_CONN_INFO udpConnInfo;
	} connInfo;

	union  
	{
		NF_TCP_CONN_INFO tcpConnInfo;
		NF_UDP_CONN_INFO udpConnInfo;
	} connInfoReal;

	char *			processName;

	unsigned int 	uid;

	NF_PROXY_SUSPEND_STATE suspended;

	bool			filteringDisabled;

	int				localSocket;
	int				remoteSocket;
	
	time_t			ioTime;
	time_t			ioTimeout;
	
	VLIST_ENTRY		completedEntry;

	VLIST_ENTRY		ovDataListFree;
	VLIST_ENTRY		ovDataListBusy;
	VLIST_ENTRY		ovDataListCompleted;

	bool			haveNewEvents;

	bool			busy;

	volatile int refCount;
	VCS	 lock;

} NFCTX, *PNFCTX;

struct PROXY_DATA
{
	NFAPI_NS NF_EventHandler * pEventHandler;
	
	VLIST_ENTRY		nfctxList;
	unsigned long	nfctxListSize;
	
	VLIST_ENTRY		nfctxCompletedList;

	PHASH_TABLE		phtNfCtx;

	int 			listenSocket;
	unsigned short 	listenPort;

	ENDPOINT_ID		nextId;

	VCS	 lock;
};


static bool	g_initialized = false;
static PROXY_DATA g_proxyData;
static int	g_nThreads = 0;
static int	g_listenPort = 8888;

static void nfctx_addRef(PNFCTX pNfCtx);
static void nfctx_release(PNFCTX pNfCtx);
static void nfctx_free(PNFCTX pNfCtx);

static PNFCTX nfproxy_findNfCtx(ENDPOINT_ID id);

static void nfproxy_addEvent(PNFCTX pNfCtx, OV_DATA * pov);

static bool nfproxy_startTcpReadLocal(PNFCTX pNfCtx);
static bool nfproxy_startTcpReadRemote(PNFCTX pNfCtx);
static bool nfproxy_startTcpWriteLocal(PNFCTX pNfCtx, const char * buf, int len);
static bool nfproxy_startTcpWriteRemote(PNFCTX pNfCtx, const char * buf, int len);

#if defined(_DEBUG) || defined(_RELEASE_LOG)
static void dbgPrintAddress(const void * addr, const char * name, ENDPOINT_ID id)
{
	if (((struct sockaddr_in *)addr)->sin_family == AF_INET)
	{
		struct sockaddr_in * pAddr = (struct sockaddr_in *)addr;
		unsigned char * pAddr_bytes = (unsigned char*)&pAddr->sin_addr;

		DbgPrint("%s[%llu] %d.%d.%d.%d:%d", 
			name, id, 
			pAddr_bytes[0], 
			pAddr_bytes[1], 
			pAddr_bytes[2], 
			pAddr_bytes[3], 
			htons(pAddr->sin_port));
	} else
	{
		struct sockaddr_in6 * pAddr = (struct sockaddr_in6 *)addr;
		unsigned short * pAddr_words = (unsigned short*)&pAddr->sin6_addr;

		DbgPrint("%s[%llu] [%x:%x:%x:%x:%x:%x:%x:%x]:%d", 
			name, id, 
			htons(pAddr_words[0]), 
			htons(pAddr_words[1]), 
			htons(pAddr_words[2]), 
			htons(pAddr_words[3]), 
			htons(pAddr_words[4]), 
			htons(pAddr_words[5]), 
			htons(pAddr_words[6]), 
			htons(pAddr_words[7]), 
			htons(pAddr->sin6_port));
	}
}
#else
#define dbgPrintAddress(a,b,c)
#endif

static void nfproxy_setNonBlocking(int s)
{
	fcntl(s, F_SETFL, fcntl(s, F_GETFL) | O_NONBLOCK);
}

static void nfproxy_copyMapAddress(sockaddr * dst, const sockaddr * src)
{
	if (src->sa_family == AF_INET6)
	{
		memcpy(dst, src, sizeof(sockaddr_in6));
	} else
	{
		memcpy((char*)&((sockaddr_in6*)dst)->sin6_addr + 6*sizeof(short), 
			&((sockaddr_in*)src)->sin_addr, 
			2*sizeof(short));
		((sockaddr_in6*)dst)->sin6_port = ((sockaddr_in*)src)->sin_port;
	}
}

static bool nfproxy_copyMappedAddress(sockaddr * dst, const sockaddr * src)
{
	static unsigned short mapToIPv6[] = { 0, 0, 0, 0, 0, 0xffff };

	if (src->sa_family == AF_INET6)
	{
		if (memcmp(&((sockaddr_in6*)src)->sin6_addr, mapToIPv6, sizeof(mapToIPv6)) == 0)
		{
			((sockaddr_in*)dst)->sin_family = AF_INET;

			memcpy(&((sockaddr_in*)dst)->sin_addr,
					(char*)&((sockaddr_in6*)src)->sin6_addr + 6*sizeof(short), 
					2*sizeof(short));
			
			((sockaddr_in*)dst)->sin_port = ((sockaddr_in6*)src)->sin6_port;

			return true;
		} else
		{
			memcpy(dst, src, sizeof(sockaddr_in6));
		}
	}

	return false;
}

static bool nfproxy_isLocalAddress(const sockaddr * pAddr)
{
	struct ifaddrs *ifaddr, *ifa;
	bool result = false;
	static unsigned short mapToIPv6[] = { 0, 0, 0, 0, 0, 0xffff };
	sockaddr_in mappedAddr;

	if (pAddr->sa_family == AF_INET6)
	{
		if (memcmp(&((sockaddr_in6*)pAddr)->sin6_addr, mapToIPv6, sizeof(mapToIPv6)) == 0)
		{
			memset(&mappedAddr, 0, sizeof(mappedAddr));
			mappedAddr.sin_family = AF_INET;
			memcpy(&mappedAddr.sin_addr, (char*)&((sockaddr_in6*)pAddr)->sin6_addr + 6*sizeof(short), 2*sizeof(short));
			pAddr = (sockaddr*)&mappedAddr;

			DbgPrint("nfproxy_isLocalAddress mapped IPv4 %x", mappedAddr.sin_addr.s_addr);
		}
	}

	if (getifaddrs(&ifaddr) == -1) 
	{
		return false;
	}

	for (ifa = ifaddr; ifa != NULL; ifa = ifa->ifa_next) 
	{
		if ((ifa->ifa_addr == NULL) || 
			(ifa->ifa_addr->sa_family != pAddr->sa_family))
			continue;

		if (pAddr->sa_family == AF_INET)
		{
			if (memcmp(&((sockaddr_in*)pAddr)->sin_addr, &((sockaddr_in*)ifa->ifa_addr)->sin_addr, 4) == 0)
			{ 
				result = true;
				break;
			}
		} else
		{
			if (memcmp(&((sockaddr_in6*)pAddr)->sin6_addr, &((sockaddr_in6*)ifa->ifa_addr)->sin6_addr, 16) == 0)
			{ 
				result = true;
				break;
			}
		}
	}

	freeifaddrs(ifaddr);
	
	return result;
}


class AutoNfCtx
{
public:
	AutoNfCtx(ENDPOINT_ID id)
	{
		if (!g_initialized)
		{
			ptr = NULL;
			return;
		}
		ptr = nfproxy_findNfCtx(id);
		if (ptr)
		{
			vcs_lock(&ptr->lock);
		}
	}
	~AutoNfCtx()
	{
		if (ptr)
		{
			vcs_unlock(&ptr->lock);
			nfctx_release(ptr);
		}
	}
	void lock()
	{
		if (ptr)
		{
			vcs_lock(&ptr->lock);
		}
	}
	void unlock()
	{
		if (ptr)
		{
			vcs_unlock(&ptr->lock);
		}
	}
	PNFCTX operator ->()
	{
		return ptr;
	}
	operator NFCTX*()
	{
		return ptr;
	}
	PNFCTX ptr;
};

static void vbufferList_free(VLIST_ENTRY * listHead)
{
	VLIST_ENTRY * pEntry;
	vbuffer_t packet;
	int n = 0;

	while (!vlist_isEmpty(listHead))
	{
		pEntry = vlist_removeHead(listHead);

		packet = vbuffer_fromListEntry(pEntry);

		DbgPrint("vbufferList_free[%d]", n++);

		vbuffer_free(packet);
	}
}

static void vbufferList_clear(VLIST_ENTRY * listHead)
{
	VLIST_ENTRY * pEntry;
	vbuffer_t packet;

	vlist_forEach(listHead, pEntry)
	{
		packet = vbuffer_fromListEntry(pEntry);

		vbuffer_clear(packet);
	}
}

static void proxyFlow_setDefaults(PROXY_FLOW * pFlow)
{
	vbufferList_free(&pFlow->packetList);
	
	pFlow->disconnected = false;
	pFlow->receiveInProgress = false;
	pFlow->sendInProgress = false;
	pFlow->disconnect = false;
	pFlow->closed = false;
	pFlow->bufferSize = NF_PACKET_SIZE;
	pFlow->packetListSize = 0;
	pFlow->eventsActive = 0;
	pFlow->shut = false;
}

static void proxyFlow_init(PROXY_FLOW * pFlow)
{
	vlist_init(&pFlow->packetList);
	proxyFlow_setDefaults(pFlow);
}

static bool proxyFlow_allocatePacket(PROXY_FLOW * pFlow, const char * buf, unsigned long len, bool toTail)
{
	vbuffer_t packet;

	if (toTail && !vlist_isEmpty(&pFlow->packetList))
	{
		PVLIST_ENTRY entry;

		entry = vlist_getTail(&pFlow->packetList);
		packet = vbuffer_fromListEntry(entry);
		
		if (vbuffer_getSize(packet) < NF_MAX_PACKET_SIZE)
		{
			if (!vbuffer_append(packet, buf, len, (buf == NULL)? false : true))
			{
				return false;
			}

			return true;
		}
	}

	packet = vbuffer_create();
	if (!packet)
		return false;

//	DbgPrint("proxyFlow_addPacket %p", packet);

	if (!vbuffer_append(packet, buf, len, (buf == NULL)? false : true))
	{
		vbuffer_free(packet);
		return false;
	}

	if (toTail)
	{
		vlist_insertTail(&pFlow->packetList, vbuffer_getListEntry(packet));
	} else
	{
		vlist_insertHead(&pFlow->packetList, vbuffer_getListEntry(packet));
	}

	pFlow->packetListSize++;

	return true;
}

static OV_DATA * ovData_alloc(int type, PNFCTX pNfCtx)
{
	OV_DATA * pov;
	VLIST_ENTRY * pEntry;

	vcs_lock(&pNfCtx->lock);

	if (vlist_isEmpty(&pNfCtx->ovDataListFree))
	{
		pov = vmalloc(OV_DATA);
		if (!pov)
		{
			vcs_unlock(&pNfCtx->lock);
			return NULL;
		}
	} else
	{
		pEntry = vlist_removeHead(&pNfCtx->ovDataListFree);
		pov = vlist_record(pEntry, OV_DATA, entry);
	}

	memset(pov, 0, sizeof(OV_DATA));

	pov->id = pNfCtx->id;
	pov->type = type;
	pov->pNfCtx = pNfCtx;

	vlist_init(&pov->packetList);

	vlist_insertTail(&pNfCtx->ovDataListBusy, &pov->entry);

	vcs_unlock(&pNfCtx->lock);

	nfctx_addRef(pNfCtx);

	return pov;
}

static void ovData_free(OV_DATA * pov)
{
	vlist_removeEntry(&pov->entry);

	vbufferList_free(&pov->packetList);

	::free(pov);
}

static void ovData_release(OV_DATA * pov, PNFCTX pNfCtx)
{
	vbufferList_free(&pov->packetList);

	vcs_lock(&pNfCtx->lock);
	vlist_removeEntry(&pov->entry);
	vlist_insertTail(&pNfCtx->ovDataListFree, &pov->entry);
	vcs_unlock(&pNfCtx->lock);

	nfctx_release(pNfCtx);
}

static bool ovData_allocatePacket(OV_DATA * pov, const char * buf, int len)
{
	vbuffer_t packet;

	packet = vbuffer_create();
	if (!packet)
	{
		DbgPrint("ovData_allocatePacket vbuffer_create failed");
		return false;
	}

	if (!vbuffer_append(packet, buf, len, (buf == NULL)? false : true))
	{
		DbgPrint("ovData_allocatePacket vbuffer_append failed");
		vbuffer_free(packet);
		return false;
	}

	vlist_insertTail(&pov->packetList, vbuffer_getListEntry(packet));

	return true;
}

static PNFCTX nfctx_alloc()
{
	PNFCTX pNfCtx;

	pNfCtx = vmalloc(NFCTX);
	if (!pNfCtx)
		return NULL;

	pNfCtx->id = 0;
	pNfCtx->protocol = IPPROTO_TCP;
	pNfCtx->proxyState = PS_NONE;
	pNfCtx->suspended = PSS_NONE;
	pNfCtx->filteringDisabled = false;
	pNfCtx->refCount = 1;
	pNfCtx->localSocket = INVALID_SOCKET;
	pNfCtx->remoteSocket = INVALID_SOCKET;
	pNfCtx->ioTime = time(NULL);
	pNfCtx->ioTimeout = 0;
	pNfCtx->processName = NULL;
	memset(&pNfCtx->connInfo, 0, sizeof(pNfCtx->connInfo));

	proxyFlow_init(&pNfCtx->localFlow);
	pNfCtx->localFlow.pNfCtx = pNfCtx;
	pNfCtx->localFlow.isLocal = true;

	proxyFlow_init(&pNfCtx->remoteFlow);
	pNfCtx->remoteFlow.pNfCtx = pNfCtx;
	pNfCtx->remoteFlow.isLocal = false;

	vlist_init(&pNfCtx->ovDataListFree);
	vlist_init(&pNfCtx->ovDataListBusy);
	vlist_init(&pNfCtx->ovDataListCompleted);

	vlist_init(&pNfCtx->entry);
	vlist_init(&pNfCtx->completedEntry);

	pNfCtx->haveNewEvents = false;
	pNfCtx->busy = false;

	if (!vcs_init(&pNfCtx->lock))
	{
		free(pNfCtx);
		return NULL;
	}

	return pNfCtx;
}

static void nfctx_setDefaults(PNFCTX pNfCtx)
{
	pNfCtx->proxyState = PS_NONE;
	pNfCtx->suspended = PSS_NONE;
	pNfCtx->filteringDisabled = false;
	memset(&pNfCtx->connInfo, 0, sizeof(pNfCtx->connInfo));

	proxyFlow_setDefaults(&pNfCtx->localFlow);
	proxyFlow_setDefaults(&pNfCtx->remoteFlow);
}

static void nfctx_addRef(PNFCTX pNfCtx)
{
	InterlockedIncrement(&pNfCtx->refCount);

#ifdef DEBUG_REFERENCES
	DbgPrint("nfctx_addRef[%llu] refCount=%d", 
		pNfCtx->id, pNfCtx->refCount);
#endif
}

static void nfctx_clearOvList(VLIST_ENTRY * list)
{
	VLIST_ENTRY * pEntry;
	OV_DATA * pov;
	
	while (!vlist_isEmpty(list))
	{
		pEntry = vlist_removeHead(list);
		pov = vlist_record(pEntry, OV_DATA, entry);

#ifdef DEBUG_REFERENCES
		DbgPrint("nfctx_clearOvList[%llu] type=%d", 
			pov->id, pov->type);
#endif
		ovData_free(pov);
	}
}

static void nfctx_releaseOvList(VLIST_ENTRY * list)
{
	VLIST_ENTRY * pEntry;
	OV_DATA * pov;
	
	while (!vlist_isEmpty(list))
	{
		pEntry = vlist_removeHead(list);
		pov = vlist_record(pEntry, OV_DATA, entry);

#ifdef DEBUG_REFERENCES
		DbgPrint("nfctx_releaseOvList[%llu] type=%d", 
			pov->id, pov->type);
#endif
		ovData_release(pov, pov->pNfCtx);
	}
}

static void nfctx_close(PNFCTX pNfCtx)
{
	DbgPrint("nfctx_close[%llu] state=%d", pNfCtx->id, pNfCtx->proxyState);

	if (pNfCtx->proxyState == PS_CLOSED)
		return;

	pNfCtx->proxyState = PS_CLOSED;

	pNfCtx->localFlow.closed = true;
	pNfCtx->remoteFlow.closed = true;

	if (pNfCtx->localSocket != INVALID_SOCKET)
	{
		close(pNfCtx->localSocket);
		pNfCtx->localSocket = INVALID_SOCKET;
	}

	if (pNfCtx->remoteSocket != INVALID_SOCKET)
	{
		close(pNfCtx->remoteSocket);
		pNfCtx->remoteSocket = INVALID_SOCKET;
	}
}

static void nfctx_release(PNFCTX pNfCtx)
{
	int refCount;

#ifdef DEBUG_REFERENCES
	DbgPrint("nfctx_release[%llu] refCount=%d", pNfCtx->id, pNfCtx->refCount);
#endif
	vcs_lock(&g_proxyData.lock);

	if (pNfCtx->refCount <= 0)
	{
		vcs_unlock(&g_proxyData.lock);
		return;
	}

	refCount = InterlockedDecrement(&pNfCtx->refCount);

	if (refCount > 0)
	{
		vcs_unlock(&g_proxyData.lock);
		return;
	}

	DbgPrint("nfctx_release[%llu] delete", pNfCtx->id);

	ht_remove_entry(g_proxyData.phtNfCtx, pNfCtx->id);

	vlist_removeEntry(&pNfCtx->entry);

	vlist_removeEntry(&pNfCtx->completedEntry);

	g_proxyData.nfctxListSize--;

	vcs_unlock(&g_proxyData.lock);

	nfctx_free(pNfCtx);
}

static void nfctx_free(PNFCTX pNfCtx)
{
	DbgPrint("nfctx_free[%llu]", pNfCtx->id);

	vbufferList_free(&pNfCtx->localFlow.packetList);

	vbufferList_free(&pNfCtx->remoteFlow.packetList);

	if (pNfCtx->processName != NULL)
	{
		free(pNfCtx->processName);
	}

	nfctx_close(pNfCtx);

	nfctx_clearOvList(&pNfCtx->ovDataListFree);
	nfctx_clearOvList(&pNfCtx->ovDataListBusy);
	nfctx_clearOvList(&pNfCtx->ovDataListCompleted);

	vcs_free(&pNfCtx->lock);

	::free(pNfCtx);
}

static PNFCTX nfproxy_addNewNfCtx()
{
	PNFCTX pNfCtx;

	pNfCtx = nfctx_alloc();
	if (!pNfCtx)
	{
		DbgPrint("nfproxy_addNewNfCtx unable to allocate a context");
		return NULL;
	}

	vcs_lock(&g_proxyData.lock);

	pNfCtx->id = g_proxyData.nextId;
	g_proxyData.nextId += 2;

	if (!ht_add_entry(g_proxyData.phtNfCtx, (PHASH_TABLE_ENTRY)&pNfCtx->id))
	{
		DbgPrint("nfproxy_addNewNfCtx[%llu] duplicated id", pNfCtx->id);
		nfctx_free(pNfCtx);
		vcs_unlock(&g_proxyData.lock);
		return NULL;
	}

	vlist_insertTail(&g_proxyData.nfctxList, &pNfCtx->entry);

	g_proxyData.nfctxListSize++;

	vcs_unlock(&g_proxyData.lock);

	return pNfCtx;
}


static void nfproxy_threadStarted()
{
	g_proxyData.pEventHandler->threadStart();
}

static void nfproxy_threadStopped()
{
	g_proxyData.pEventHandler->threadEnd();
}

static PNFCTX nfproxy_findNfCtx(ENDPOINT_ID id)
{
	PHASH_TABLE_ENTRY pEntry;
	PNFCTX pNfCtx;

	vcs_lock(&g_proxyData.lock);
	
	pEntry = ht_find_entry(g_proxyData.phtNfCtx, id);
	if (pEntry)
	{
		pNfCtx = vlist_record(pEntry, NFCTX, id);

		nfctx_addRef(pNfCtx);

		vcs_unlock(&g_proxyData.lock);

		return pNfCtx;
	}
	
	vcs_unlock(&g_proxyData.lock);

	return NULL;
}

static void nfproxy_postClose(PNFCTX pNfCtx)
{
	OV_DATA * newov = ovData_alloc(OVT_TCP_CLOSED, pNfCtx);
	if (newov)
	{
		nfproxy_addEvent(pNfCtx, newov);
	} else
	{
		nfctx_release(pNfCtx);
	}
}

static void nfproxy_onTcpAccepted(OV_DATA * pov)
{
	PNFCTX   pNfCtx;
	int s;
	int err;

	DbgPrint("nfproxy_onTcpAccepted[%llu]", pov->id);

	pNfCtx = pov->pNfCtx;

	for (;;)
	{
		sockaddr_in6 addr;

		memcpy(&addr, pNfCtx->connInfoReal.tcpConnInfo.remoteAddress, sizeof(addr));
	
		dbgPrintAddress(&addr, "connect to ", pov->id);

		s = socket(AF_INET6, SOCK_STREAM, 0);
		if (s == INVALID_SOCKET)
			break;

		pNfCtx->remoteSocket = s;

		int optVal = 1;

	 	if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR, &optVal, sizeof(optVal)) == -1)
		{
			DbgPrint("nfproxy_onTcpAccepted socket %d, SO_REUSEADDR error %d", s, errno);
        	break;
	    }

		optVal = 1;
	 	if (setsockopt(s, IPPROTO_TCP, TCP_NODELAY, &optVal, sizeof(optVal)) == -1)
		{
			DbgPrint("nfproxy_onTcpAccepted socket %d, TCP_NODELAY error %d", s, errno);
        	break;
	    }

		optVal = MARK_OUT;
	 	if (setsockopt(s, SOL_SOCKET, SO_MARK, &optVal, sizeof(optVal)) == -1)
		{
			DbgPrint("nfproxy_onTcpAccepted SO_MARK error %d", errno);
        	break;
	    }

		nfproxy_setNonBlocking(s);

		pNfCtx->connInfo.tcpConnInfo.filteringFlag = NF_FILTER;

		{
			TCP_PROC_INFO tpi;

			if (pNfCtx->connInfo.tcpConnInfo.direction == NF_D_OUT)
			{
				if (nfutil_getTCPProcInfo(
						ntohs(((sockaddr_in6*)pNfCtx->connInfoReal.tcpConnInfo.localAddress)->sin6_port), 
						&tpi))
				{
					pNfCtx->connInfo.tcpConnInfo.processId = tpi.pid;
					pNfCtx->uid = tpi.uid;
				}

				vcs_unlock(&pNfCtx->lock);
				g_proxyData.pEventHandler->tcpConnectRequest(pNfCtx->id, &pNfCtx->connInfo.tcpConnInfo);
				vcs_lock(&pNfCtx->lock);
		
				if (pNfCtx->connInfo.tcpConnInfo.filteringFlag & NF_BLOCK)
				{
					break;
				}

				nfproxy_copyMapAddress(
					(sockaddr*)&addr, 
					(sockaddr*)pNfCtx->connInfo.tcpConnInfo.remoteAddress);

				dbgPrintAddress(&addr, "connect to ", pov->id);
			} else
			{
				if (nfutil_getTCPProcInfo(
						ntohs(((sockaddr_in6*)pNfCtx->connInfoReal.tcpConnInfo.remoteAddress)->sin6_port), 
						&tpi))
				{
					pNfCtx->connInfo.tcpConnInfo.processId = tpi.pid;
					pNfCtx->uid = tpi.uid;
				}

				if (pNfCtx->connInfo.tcpConnInfo.processId == getpid())
					break;

				memcpy(pNfCtx->connInfo.tcpConnInfo.remoteAddress, 
					pNfCtx->connInfo.tcpConnInfo.localAddress, 
					NF_MAX_ADDRESS_LENGTH);
				
				nfproxy_copyMappedAddress(
					(sockaddr*)pNfCtx->connInfo.tcpConnInfo.localAddress,
					(sockaddr*)&addr
				);
			}
		}

		err = connect(s, (sockaddr*)&addr, sizeof(addr)); 
		if (err != 0)
		{
			if (errno != EINPROGRESS)
			{
				DbgPrint("nfproxy_onTcpAccepted connect error %d", errno);
				break;
			}
		}
	
		pNfCtx->ioTimeout = NF_CONNECT_TIMEOUT;

		OV_DATA * opData = ovData_alloc(OVT_TCP_CONNECTED, pNfCtx);
		if (!opData)
		{
			DbgPrint("nfproxy_onTcpAccepted ovData_alloc error %d", errno);
			break;
		}

		if (!nfiocp_registerSocket(s, pNfCtx->id + 1, EPOLLIN | EPOLLOUT | EPOLLRDHUP | EPOLLPRI | EPOLLET))
		{
			DbgPrint("nfproxy_onTcpAccepted nfiocp_registerSocket error %d", errno);
			break;
		}

		return;
	}

	DbgPrint("nfproxy_onTcpAccepted error %d", errno);

	nfproxy_postClose(pNfCtx);
}

static void nfproxy_onTcpConnected(OV_DATA * pov)
{
	PNFCTX pNfCtx;

	pNfCtx = pov->pNfCtx;

	if (pov->error != 0)
	{
		DbgPrint("nfproxy_onTcpConnected[%llu] error=%d", pNfCtx->id, pov->error);
		nfproxy_postClose(pNfCtx);
		return;
	}

	DbgPrint("nfproxy_onTcpConnected[%llu]", pNfCtx->id);

	nfproxy_setNonBlocking(pNfCtx->localSocket);

	if (!nfiocp_registerSocket(pNfCtx->localSocket, pNfCtx->id, EPOLLIN | EPOLLOUT | EPOLLRDHUP | EPOLLPRI | EPOLLET))
	{
		DbgPrint("nfproxy_onTcpConnected nfiocp_registerSocket error %d", errno);
		nfproxy_postClose(pNfCtx);
		return;
	}

	if (pNfCtx->connInfo.tcpConnInfo.filteringFlag & NF_FILTER)
	{
		vcs_unlock(&pNfCtx->lock);
		g_proxyData.pEventHandler->tcpConnected(pNfCtx->id, &pNfCtx->connInfo.tcpConnInfo);
		vcs_lock(&pNfCtx->lock);
	}

	pNfCtx->proxyState = PS_CONNECTED;

	pNfCtx->ioTimeout = 0;
	
	pNfCtx->localFlow.eventsActive |= EPOLLIN | EPOLLOUT;
	pNfCtx->remoteFlow.eventsActive |= EPOLLIN | EPOLLOUT;
	
	nfproxy_startTcpReadLocal(pNfCtx);
	nfproxy_startTcpReadRemote(pNfCtx);
}

static void nfproxy_onClosed(OV_DATA * pov)
{
	PNFCTX   pNfCtx;

	DbgPrint("nfproxy_onClosed[%llu]", pov->id);

	pNfCtx = pov->pNfCtx;

	if (pNfCtx->proxyState == PS_CLOSED)
		return;

	nfctx_close(pNfCtx);

	nfctx_releaseOvList(&pNfCtx->ovDataListBusy);
	nfctx_releaseOvList(&pNfCtx->ovDataListCompleted);

	if (pNfCtx->connInfo.tcpConnInfo.filteringFlag & NF_FILTER)
	{
		vcs_unlock(&pNfCtx->lock);
		g_proxyData.pEventHandler->tcpClosed(pNfCtx->id, &pNfCtx->connInfo.tcpConnInfo);
		vcs_lock(&pNfCtx->lock);
	}

	nfctx_release(pNfCtx);
}

static bool nfproxy_recv(int socket, OV_DATA * pov)
{
	int ret;
	vbuffer_t buffer;

	if (vlist_isEmpty(&pov->packetList))
	{
		DbgPrint("nfproxy_recv[%llu] no data", 
				pov->id);
		return false;
	}

	buffer = vbuffer_fromListEntry(vlist_getHead(&pov->packetList));
	if (vbuffer_getSize(buffer) == 0)	
	{
		DbgPrint("nfproxy_recv[%llu] zero length buffer", 
				pov->id);
		return false;
	}

	for (;;)
	{
		DbgPrint("nfproxy_recv[%llu]", pov->id);

		ret = recv(socket, vbuffer_buffer(buffer), (size_t)vbuffer_getSize(buffer), MSG_DONTWAIT);
		if (ret < 0)
		{
			int error = errno;

			DbgPrint("nfproxy_recv[%llu] error=%d", 
					pov->id, error);

			if (error == EINTR)
			{	
				continue;
			}

			if (error == EAGAIN ||
				error == EWOULDBLOCK)
			{
				return false;
			}

			pov->error = -1;
			pov->transferred = 0;
			break;
		} else
		{
			pov->error = 0;
			pov->transferred = ret;

			DbgPrint("nfproxy_recv[%llu] returned %d", 
					pov->id, ret);
		}

		break;
	}

	return true;
}

static bool nfproxy_send(int socket, OV_DATA * pov)
{
	int ret;
	vbuffer_t buffer;

	if (vlist_isEmpty(&pov->packetList))
	{
		DbgPrint("nfproxy_send[%llu] no data", 
				pov->id);
		return false;
	}

	buffer = vbuffer_fromListEntry(vlist_getHead(&pov->packetList));
	if (vbuffer_getSize(buffer) == 0)	
	{
		DbgPrint("nfproxy_send[%llu] zero length buffer", 
				pov->id);
		return false;
	}

	for (;;)
	{
		DbgPrint("nfproxy_send[%llu]", pov->id);

		ret = send(socket, 
			vbuffer_buffer(buffer), 
			(size_t)vbuffer_getSize(buffer), 
			MSG_NOSIGNAL);

		if (ret < 0)
		{
			int error = errno;

			DbgPrint("nfproxy_send[%llu] error=%d", 
					pov->id, error);

			if (error == EINTR)
			{	
				continue;
			}

			if (error == EAGAIN ||
				error == EWOULDBLOCK)
			{
				return false;
			}

			pov->error = -1;
			pov->transferred = 0;
			break;
		} else
		{
			pov->error = 0;
			pov->transferred = ret;
		}

		break;
	}

	return true;
}

static void nfproxy_onTcpLocalEvent(OV_DATA * pov)
{
	PNFCTX pNfCtx = pov->pNfCtx;
	int events = (int)pov->transferred;
	int eventsActive;
	VLIST_ENTRY * pEntry;
	OV_DATA * opData;

	DbgPrint("nfproxy_onTcpLocalEvent[%llu] events=0x%x", 
				pov->id, events);

	if (pNfCtx->proxyState == PS_CLOSED)
		return;

	pNfCtx->localFlow.eventsActive |= events;
	eventsActive = pNfCtx->localFlow.eventsActive;

	for (pEntry = vlist_getHead(&pNfCtx->ovDataListBusy); 
        pEntry != &pNfCtx->ovDataListBusy; 
        )
	{
		opData = vlist_record(pEntry, OV_DATA, entry);

		pEntry = vlist_getHead(pEntry);

		switch (opData->type)
		{
		case OVT_TCP_LOCAL_READ:
			{
				if (!(eventsActive & EPOLLIN))
				{
					if (eventsActive & (EPOLLERR | EPOLLHUP | EPOLLRDHUP | EPOLLPRI))
					{
						opData->error = -1;
						nfproxy_addEvent(pNfCtx, opData);
					} 

					break;
				}

				if (nfproxy_recv(pNfCtx->localSocket, opData))
				{
					if (opData->transferred < 
						vbuffer_getSize(vbuffer_fromListEntry(vlist_getHead(&opData->packetList))))
					{
						pNfCtx->localFlow.eventsActive &= ~EPOLLIN;
					}

					if (opData->transferred == 0)
					{
						eventsActive |= EPOLLHUP;
					}

					nfproxy_addEvent(pNfCtx, opData);
				} else
				{
					pNfCtx->localFlow.eventsActive &= ~EPOLLIN;
				}

				break;
			}

		case OVT_TCP_LOCAL_WRITE:
			{
				if (!(eventsActive & EPOLLOUT))
				{
					if (eventsActive & (EPOLLERR | EPOLLHUP | EPOLLRDHUP | EPOLLPRI))
					{
						opData->error = -1;
						nfproxy_addEvent(pNfCtx, opData);
					} 

					break;
				}

				if (nfproxy_send(pNfCtx->localSocket, opData))
				{
					vbuffer_t buf = vbuffer_fromListEntry(vlist_getHead(&opData->packetList));

					if (opData->transferred < vbuffer_getSize(buf))
					{
						pNfCtx->localFlow.eventsActive &= ~EPOLLOUT;

						DbgPrint("nfproxy_onTcpLocalEvent[%llu] partial send sent=%d total=%d", 
							pov->id, opData->transferred, vbuffer_getSize(buf));

						if (!proxyFlow_allocatePacket(&pNfCtx->localFlow, 
								vbuffer_buffer(buf) + opData->transferred, 
								vbuffer_getSize(buf) - opData->transferred,
								false))
						{
							DbgPrint("nfproxy_onTcpLocalEvent[%llu] failed to process partial send", 
								pov->id);
						}
					}

					nfproxy_addEvent(pNfCtx, opData);
				} else
				{
					pNfCtx->localFlow.eventsActive &= ~EPOLLOUT;
				}

				break;
			}
		}
	}
}

static void nfproxy_onTcpRemoteEvent(OV_DATA * pov)
{
	PNFCTX pNfCtx = pov->pNfCtx;
	int events = (int)pov->transferred;
	int eventsActive;
	VLIST_ENTRY * pEntry;
	OV_DATA * opData;

	if (pNfCtx->proxyState == PS_CLOSED)
		return;

	DbgPrint("nfproxy_onTcpRemoteEvent[%llu] events=0x%x", 
				pov->id, events);

	pNfCtx->remoteFlow.eventsActive |= events;
	eventsActive = pNfCtx->remoteFlow.eventsActive;

	for (pEntry = vlist_getHead(&pNfCtx->ovDataListBusy); 
        pEntry != &pNfCtx->ovDataListBusy; 
        )
	{
		opData = vlist_record(pEntry, OV_DATA, entry);

		pEntry = vlist_getHead(pEntry);

		switch (opData->type)
		{
		case OVT_TCP_CONNECTED:
			{
				if (!(eventsActive & EPOLLOUT))
					break;

				if (eventsActive & (EPOLLERR | EPOLLHUP))
					opData->error = -1;

				nfproxy_addEvent(pNfCtx, opData);

				break;
			}

		case OVT_TCP_REMOTE_READ:
			{
				if (!(eventsActive & EPOLLIN))
				{
					if (eventsActive & (EPOLLERR | EPOLLHUP | EPOLLRDHUP | EPOLLPRI))
					{
						opData->error = -1;
						nfproxy_addEvent(pNfCtx, opData);
					} 

					break;
				}

				if (nfproxy_recv(pNfCtx->remoteSocket, opData))
				{
					if (opData->transferred < 
						vbuffer_getSize(vbuffer_fromListEntry(vlist_getHead(&opData->packetList))))
					{
						pNfCtx->remoteFlow.eventsActive &= ~EPOLLIN;
					}

					if (opData->transferred == 0)
					{
						eventsActive |= EPOLLHUP;
					}

					nfproxy_addEvent(pNfCtx, opData);
				} else
				{
					pNfCtx->remoteFlow.eventsActive &= ~EPOLLIN;
				}

				break;
			}

		case OVT_TCP_REMOTE_WRITE:
			{
				if (!(eventsActive & EPOLLOUT))
				{
					if (eventsActive & (EPOLLERR | EPOLLHUP | EPOLLRDHUP | EPOLLPRI))
					{
						opData->error = -1;
						nfproxy_addEvent(pNfCtx, opData);
					}

					break;
				}

				if (nfproxy_send(pNfCtx->remoteSocket, opData))
				{
					vbuffer_t buf = vbuffer_fromListEntry(vlist_getHead(&opData->packetList));

					if (opData->transferred < vbuffer_getSize(buf))
					{
						pNfCtx->remoteFlow.eventsActive &= ~EPOLLOUT;

						DbgPrint("nfproxy_onTcpRemoteEvent[%llu] partial send sent=%d total=%d", 
							pov->id, opData->transferred, vbuffer_getSize(buf));

						if (!proxyFlow_allocatePacket(&pNfCtx->remoteFlow, 
								vbuffer_buffer(buf) + opData->transferred, 
								vbuffer_getSize(buf) - opData->transferred,
								false))
						{
							DbgPrint("nfproxy_onTcpRemoteEvent[%llu] failed to process partial send", 
								pov->id);
						}
					}

					nfproxy_addEvent(pNfCtx, opData);
				} else
				{
					pNfCtx->remoteFlow.eventsActive &= ~EPOLLOUT;
				}

				break;
			}
		}
	}
}

static void nfproxy_onTcpLocalWriteComplete(OV_DATA * pov)
{
	PNFCTX   pNfCtx;
	PROXY_FLOW * psd;

	DbgPrint("nfproxy_onTcpLocalWriteComplete[%llu] bytes=%d, error=%d", 
		pov->id, pov->transferred, pov->error);

	pNfCtx = pov->pNfCtx;

	if (pNfCtx->localSocket == INVALID_SOCKET ||
		pNfCtx->remoteSocket == INVALID_SOCKET)
	{
		return;
	}

	psd = &pNfCtx->localFlow;
	psd->sendInProgress = false;

	if (pov->transferred == 0)
	{
		DbgPrint("nfproxy_onTcpLocalWriteComplete[%llu] error=%d", 
			pov->id, pov->error);

		if (pov->error != 0)
		{
			shutdown(pNfCtx->remoteSocket, SHUT_RD);
			vbufferList_free(&psd->packetList);
		}

		nfproxy_startTcpWriteLocal(pNfCtx, NULL, -1);

		nfproxy_startTcpReadRemote(pNfCtx);
		return;
	}

	if (psd->disconnect || !vlist_isEmpty(&psd->packetList))
	{
		nfproxy_startTcpWriteLocal(pNfCtx, NULL, -1);
		nfproxy_startTcpReadRemote(pNfCtx);
	} else
	if (pNfCtx->proxyState == PS_CONNECTED)
	{
		nfproxy_startTcpReadRemote(pNfCtx);

		if (!pNfCtx->filteringDisabled)
		{
			vcs_unlock(&pNfCtx->lock);
			g_proxyData.pEventHandler->tcpCanReceive(pov->id);
			vcs_lock(&pNfCtx->lock);
		}
	}
}

static void nfproxy_onTcpLocalReadComplete(OV_DATA * pov)
{
	PNFCTX   pNfCtx;
	PROXY_FLOW * psd;

	DbgPrint("nfproxy_onTcpLocalReadComplete[%llu] bytes=%d, error=%d", 
		pov->id, pov->transferred, pov->error);

	pNfCtx = pov->pNfCtx;

	if (pNfCtx->localSocket == INVALID_SOCKET)
	{
		return;
	}

	psd = &pNfCtx->localFlow;

	psd->receiveInProgress = false;

	if (pov->transferred == 0)
	{
		if (pov->error != 0)
		{
			DbgPrint("nfproxy_onTcpLocalReadComplete[%llu] error=%d", 
				pov->id, pov->error);
			
			psd->closed = true;
		}

		if (pNfCtx->proxyState == PS_CONNECTED &&
			pNfCtx->connInfo.tcpConnInfo.filteringFlag & NF_FILTER)
		{
			if (!pNfCtx->localFlow.disconnected)
			{
				pNfCtx->localFlow.disconnected = true;

				if (!pNfCtx->filteringDisabled)
				{
					vcs_unlock(&pNfCtx->lock);
					if (pNfCtx->connInfo.tcpConnInfo.direction == NF_D_IN)
					{
						g_proxyData.pEventHandler->tcpReceive(pov->id, NULL, 0);
					} else
					{
						g_proxyData.pEventHandler->tcpSend(pov->id, NULL, 0);
					}
					vcs_lock(&pNfCtx->lock);
				} else
				{
					nfproxy_startTcpWriteRemote(pNfCtx, NULL, 0);
				}
			}
		} else
		{
			pNfCtx->localFlow.disconnected = true;
			nfproxy_startTcpWriteRemote(pNfCtx, NULL, 0);
		}

		return;
	}

	if (pNfCtx->proxyState == PS_CONNECTED)
	{
		char * buf = vbuffer_buffer(
			vbuffer_fromListEntry(vlist_getHead(&pov->packetList)));

		if (psd->bufferSize == pov->transferred)
		{
			if (psd->bufferSize < NF_MAX_PACKET_SIZE)
			{
				psd->bufferSize *= 2;
			}
		}

		if (pNfCtx->connInfo.tcpConnInfo.filteringFlag & NF_FILTER &&
			!pNfCtx->filteringDisabled)
		{
			vcs_unlock(&pNfCtx->lock);
			if (pNfCtx->connInfo.tcpConnInfo.direction == NF_D_IN)
			{
				g_proxyData.pEventHandler->tcpReceive(pov->id, buf, pov->transferred);
			} else
			{
				g_proxyData.pEventHandler->tcpSend(pov->id, buf, pov->transferred);
			}
			vcs_lock(&pNfCtx->lock);
		} else
		{
			if (!nfproxy_startTcpWriteRemote(pNfCtx, buf, pov->transferred))
			{
				DbgPrint("nfproxy_onTcpLocalReadComplete nfproxy_startTcpWriteRemote failed");
			}
		}
	
		nfproxy_startTcpReadLocal(pNfCtx);
	} 
}

static void nfproxy_onTcpRemoteWriteComplete(OV_DATA * pov)
{
	PNFCTX   pNfCtx;
	PROXY_FLOW * psd;

	DbgPrint("nfproxy_onTcpRemoteWriteComplete[%llu] bytes=%d, error=%d", 
		pov->id, pov->transferred, pov->error);

	pNfCtx = pov->pNfCtx;

	if (pNfCtx->remoteSocket == INVALID_SOCKET)
	{
		return;
	}

	psd = &pNfCtx->remoteFlow;
	psd->sendInProgress = false;

	if (pov->transferred == 0)
	{
		DbgPrint("nfproxy_onTcpRemoteWriteComplete[%llu] error=%d", pov->id, pov->error);
			
		if (pov->error != 0)
		{
			shutdown(pNfCtx->localSocket, SHUT_RD);
			vbufferList_free(&psd->packetList);
		}

		nfproxy_startTcpWriteRemote(pNfCtx, NULL, -1);

		nfproxy_startTcpReadLocal(pNfCtx);
		return;
	}

	if (psd->disconnect || !vlist_isEmpty(&psd->packetList))
	{
		nfproxy_startTcpWriteRemote(pNfCtx, NULL, -1);
		nfproxy_startTcpReadLocal(pNfCtx);
	} else
	if (pNfCtx->proxyState == PS_CONNECTED)
	{
		nfproxy_startTcpReadLocal(pNfCtx);

		if (!pNfCtx->filteringDisabled)
		{
			vcs_unlock(&pNfCtx->lock);
			g_proxyData.pEventHandler->tcpCanSend(pov->id);
			vcs_lock(&pNfCtx->lock);
		}
	}
}

static void nfproxy_onTcpRemoteReadComplete(OV_DATA * pov)
{
	PNFCTX   pNfCtx;
	PROXY_FLOW * psd;

	DbgPrint("nfproxy_onTcpRemoteReadComplete[%llu] bytes=%d, error=%d", 
		pov->id, pov->transferred, pov->error);

	pNfCtx = pov->pNfCtx;

	if (pNfCtx->remoteSocket == INVALID_SOCKET)
	{
		return;
	}

	psd = &pNfCtx->remoteFlow;

	psd->receiveInProgress = false;

	if (pov->transferred == 0)
	{
		if (pov->error != 0)
		{
			DbgPrint("nfproxy_onTcpRemoteReadComplete[%llu] error=%d", pov->id, pov->error);
			psd->closed = true;
		}

		if (pNfCtx->proxyState == PS_CONNECTED &&
			pNfCtx->connInfo.tcpConnInfo.filteringFlag & NF_FILTER)
		{
			if (!pNfCtx->remoteFlow.disconnected)
			{
				pNfCtx->remoteFlow.disconnected = true;

				if (!pNfCtx->filteringDisabled)
				{
					vcs_unlock(&pNfCtx->lock);
					if (pNfCtx->connInfo.tcpConnInfo.direction == NF_D_IN)
					{
						g_proxyData.pEventHandler->tcpSend(pov->id, NULL, 0);
					} else
					{
						g_proxyData.pEventHandler->tcpReceive(pov->id, NULL, 0);
					}
					vcs_lock(&pNfCtx->lock);
				} else
				{
					nfproxy_startTcpWriteLocal(pNfCtx, NULL, 0);
				}
			}
		} else
		{
			pNfCtx->remoteFlow.disconnected = true;
			nfproxy_startTcpWriteLocal(pNfCtx, NULL, 0);
		}

		return;
	}

	if (pNfCtx->proxyState == PS_CONNECTED)
	{
		char * buf = vbuffer_buffer(
			vbuffer_fromListEntry(vlist_getHead(&pov->packetList)));

		if (psd->bufferSize == pov->transferred)
		{
			if (psd->bufferSize < NF_MAX_PACKET_SIZE)
			{
				psd->bufferSize *= 2;
			}
		}

		if (pNfCtx->connInfo.tcpConnInfo.filteringFlag & NF_FILTER &&
			!pNfCtx->filteringDisabled)
		{
			vcs_unlock(&pNfCtx->lock);
			if (pNfCtx->connInfo.tcpConnInfo.direction == NF_D_IN)
			{
				g_proxyData.pEventHandler->tcpSend(pov->id, buf, pov->transferred);
			} else
			{
				g_proxyData.pEventHandler->tcpReceive(pov->id, buf, pov->transferred);
			}
			vcs_lock(&pNfCtx->lock);
		} else
		{
			if (!nfproxy_startTcpWriteLocal(pNfCtx, buf, pov->transferred))
			{
				DbgPrint("nfproxy_onTcpRemoteReadComplete nfproxy_startTcpWriteLocal failed");
			}
		}

		nfproxy_startTcpReadRemote(pNfCtx);
	}
}

static bool nfproxy_startTcpReadLocal(PNFCTX pNfCtx)
{
	PROXY_FLOW * psd;

	DbgPrint("nfproxy_startTcpReadLocal[%llu]", pNfCtx->id);

	psd = &pNfCtx->localFlow;

	if (psd->closed || 
		pNfCtx->localSocket == INVALID_SOCKET)
	{
		DbgPrint("nfproxy_startTcpReadLocal[%llu] socket disconnected", pNfCtx->id);
		return false;
	}

	if (psd->receiveInProgress)
	{
		DbgPrint("nfproxy_startTcpReadLocal[%llu] receiveInProgress", pNfCtx->id);
		return true;
	}

	if ((pNfCtx->suspended & (PSS_SUSPEND | PSS_SUSPEND_OUTBOUND)) && 
		pNfCtx->proxyState == PS_CONNECTED &&
		!psd->disconnect)
	{
		DbgPrint("nfproxy_startTcpReadLocal[%llu] suspended", pNfCtx->id);
		return true;
	}

	size_t outLimit = TCP_MAX_QUEUE_SIZE_OUT;

	if (pNfCtx->remoteFlow.packetListSize > outLimit)
	{
		DbgPrint("nfproxy_startTcpReadLocal[%llu] out buffer is too large", pNfCtx->id);
		return true;
	}

	OV_DATA * pov;
	vbuffer_t packet;

	pov = ovData_alloc(OVT_TCP_LOCAL_READ, pNfCtx);
	if (!pov)
	{
		DbgPrint("nfproxy_startTcpReadLocal[%llu] ovData_alloc failed", pNfCtx->id);
		return false;
	}

	if (!ovData_allocatePacket(pov, NULL, psd->bufferSize))
	{
		DbgPrint("nfproxy_startTcpReadLocal[%llu] ovData_allocatePacket failed", pNfCtx->id);
		return false;
	}

	psd->receiveInProgress = true;

	if (psd->eventsActive != 0)
	{
		pov->transferred = 0;
		nfproxy_onTcpLocalEvent(pov);
	}

	return true;
}

static bool nfproxy_startTcpWriteLocal(PNFCTX pNfCtx, const char * buf, int len)
{
	PROXY_FLOW * psd;

	DbgPrint("nfproxy_startTcpWriteLocal[%llu]", pNfCtx->id);

	psd = &pNfCtx->localFlow;

	if (pNfCtx->localSocket == INVALID_SOCKET)
	{
		return false;
	}

	if (len == 0)
	{
		psd->disconnect = true;
	} else
	if (len != -1)
	{
		if (!proxyFlow_allocatePacket(psd, buf, len, true))
		{
			DbgPrint("nfproxy_startTcpWriteLocal[%llu] unable to allocate packet", pNfCtx->id);
			return false;
		}
	}

	if (psd->sendInProgress)
	{
		DbgPrint("nfproxy_startTcpWriteLocal[%llu] send in progress", pNfCtx->id);
		return true;
	}

	if (vlist_isEmpty(&psd->packetList))
	{
		DbgPrint("nfproxy_startTcpWriteLocal[%llu] packet list empty", pNfCtx->id);

		if (psd->disconnect)
		{
			DbgPrint("nfproxy_startTcpWriteLocal[%llu] shutdown", pNfCtx->id);

			pNfCtx->ioTimeout = NF_FIN_WAIT_TIMEOUT;
			
			pNfCtx->localFlow.shut = true;

			shutdown(pNfCtx->localSocket, SHUT_WR);
			
			nfproxy_startTcpReadLocal(pNfCtx);
		}

		if ((pNfCtx->localFlow.shut) && 
			(pNfCtx->remoteFlow.shut))
		{
			nfproxy_postClose(pNfCtx);
		}

		return true;
	}

	OV_DATA * pov;
	VLIST_ENTRY * pEntry;
	vbuffer_t packet;

	pov = ovData_alloc(OVT_TCP_LOCAL_WRITE, pNfCtx);
	if (!pov)
	{
		DbgPrint("nfproxy_startTcpWriteLocal[%llu] ovData_alloc failed", pNfCtx->id);
		return false;
	}

	pEntry = vlist_removeHead(&psd->packetList);
	packet = vbuffer_fromListEntry(pEntry);
	vlist_insertTail(&pov->packetList, pEntry);
	psd->packetListSize--;

	psd->sendInProgress = true;

	if (psd->eventsActive != 0)
	{
		pov->transferred = 0;
		nfproxy_onTcpLocalEvent(pov);
	}

	return true;
}
	
static bool nfproxy_startTcpReadRemote(PNFCTX pNfCtx)
{
	PROXY_FLOW * psd;

	DbgPrint("nfproxy_startTcpReadRemote[%llu]", pNfCtx->id);

	psd = &pNfCtx->remoteFlow;

	if (psd->closed || 
		pNfCtx->remoteSocket == INVALID_SOCKET)
	{
		DbgPrint("nfproxy_startTcpReadRemote[%llu] socket disconnected", pNfCtx->id);
		return false;
	}

	if (psd->receiveInProgress)
	{
		DbgPrint("nfproxy_startTcpReadRemote[%llu] receiveInProgress", pNfCtx->id);
		return true;
	}

	if ((pNfCtx->suspended & (PSS_SUSPEND | PSS_SUSPEND_OUTBOUND)) && 
		pNfCtx->proxyState == PS_CONNECTED &&
		!psd->disconnect)
	{
		DbgPrint("nfproxy_startTcpReadRemote[%llu] suspended", pNfCtx->id);
		return true;
	}

	size_t inLimit = TCP_MAX_QUEUE_SIZE_IN;

	if (pNfCtx->localFlow.packetListSize > inLimit)
	{
		DbgPrint("nfproxy_startTcpReadRemote[%llu] in buffer is too large", pNfCtx->id);
		return true;
	}

	OV_DATA * pov;
	vbuffer_t packet;

	pov = ovData_alloc(OVT_TCP_REMOTE_READ, pNfCtx);
	if (!pov)
	{
		DbgPrint("nfproxy_startTcpReadRemote[%llu] ovData_alloc failed", pNfCtx->id);
		return false;
	}

	if (!ovData_allocatePacket(pov, NULL, psd->bufferSize))
	{
		DbgPrint("nfproxy_startTcpReadRemote[%llu] ovData_allocatePacket failed", pNfCtx->id);
		return false;
	}

	psd->receiveInProgress = true;

	if (psd->eventsActive != 0)
	{
		pov->transferred = 0;
		nfproxy_onTcpRemoteEvent(pov);
	}

	return true;
}

static bool nfproxy_startTcpWriteRemote(PNFCTX pNfCtx, const char * buf, int len)
{
	PROXY_FLOW * psd;

	DbgPrint("nfproxy_startTcpWriteRemote[%llu]", pNfCtx->id);

	psd = &pNfCtx->remoteFlow;

	if (pNfCtx->remoteSocket == INVALID_SOCKET)
	{
		return false;
	}

	if (len == 0)
	{
		psd->disconnect = true;
	} else
	if (len != -1)
	{
		if (!proxyFlow_allocatePacket(psd, buf, len, true))
		{
			DbgPrint("nfproxy_startTcpWriteRemote[%llu] unable to allocate packet", pNfCtx->id);
			return false;
		}
	}

	if (psd->sendInProgress)
	{
		DbgPrint("nfproxy_startTcpWriteRemote[%llu] send in progress", pNfCtx->id);
		return true;
	}

	if (vlist_isEmpty(&psd->packetList))
	{
		DbgPrint("nfproxy_startTcpWriteRemote[%llu] packet list empty", pNfCtx->id);

		if (psd->disconnect)
		{
			DbgPrint("nfproxy_startTcpWriteRemote[%llu] shutdown", pNfCtx->id);

			pNfCtx->ioTimeout = NF_FIN_WAIT_TIMEOUT;

			pNfCtx->remoteFlow.shut = true;

			shutdown(pNfCtx->remoteSocket, SHUT_WR);

			nfproxy_startTcpReadRemote(pNfCtx);
		}

		if ((pNfCtx->localFlow.shut) && 
			(pNfCtx->remoteFlow.shut))
		{
			nfproxy_postClose(pNfCtx);
		}

		return true;
	}

	OV_DATA * pov;
	VLIST_ENTRY * pEntry;
	vbuffer_t packet;

	pov = ovData_alloc(OVT_TCP_REMOTE_WRITE, pNfCtx);
	if (!pov)
	{
		DbgPrint("nfproxy_startTcpWriteRemote[%llu] ovData_alloc failed", pNfCtx->id);
		return false;
	}

	pEntry = vlist_removeHead(&psd->packetList);
	packet = vbuffer_fromListEntry(pEntry);
	vlist_insertTail(&pov->packetList, pEntry);
	psd->packetListSize--;

	psd->sendInProgress = true;

	if (psd->eventsActive != 0)
	{
		pov->transferred = 0;
		nfproxy_onTcpRemoteEvent(pov);
	}

	return true;
}

NFAPI_API NF_STATUS NFAPI_NS
nf_tcpPostSend(ENDPOINT_ID id, const char * buf, int len)
{
	AutoNfCtx pNfCtx(id);
	if (!pNfCtx)
		return NF_STATUS_FAIL;

	DbgPrint("nf_tcpPostSend[%llu], len=%d", id, len);

	if (pNfCtx->connInfo.tcpConnInfo.direction == NF_D_IN)	
	{
		return nfproxy_startTcpWriteLocal(pNfCtx, buf, len)? NF_STATUS_SUCCESS : NF_STATUS_FAIL;
	} else
	{
		return nfproxy_startTcpWriteRemote(pNfCtx, buf, len)? NF_STATUS_SUCCESS : NF_STATUS_FAIL;
	}
}

NFAPI_API NF_STATUS NFAPI_NS
nf_tcpPostReceive(ENDPOINT_ID id, const char * buf, int len)
{
	AutoNfCtx pNfCtx(id);
	if (!pNfCtx)
		return NF_STATUS_FAIL;

	DbgPrint("nf_tcpPostReceive[%llu], len=%d", id, len);

	if (pNfCtx->connInfo.tcpConnInfo.direction == NF_D_IN)	
	{
		return nfproxy_startTcpWriteRemote(pNfCtx, buf, len)? NF_STATUS_SUCCESS : NF_STATUS_FAIL;
	} else
	{
		return nfproxy_startTcpWriteLocal(pNfCtx, buf, len)? NF_STATUS_SUCCESS : NF_STATUS_FAIL;
	}
}

NFAPI_API NF_STATUS NFAPI_NS
nf_tcpSetConnectionState(ENDPOINT_ID id, int suspended)
{
	AutoNfCtx pNfCtx(id);
	if (!pNfCtx)
		return NF_STATUS_FAIL;

	DbgPrint("nf_tcpSetConnectionState[%llu], suspended=%d", id, suspended);

	pNfCtx->suspended = (NF_PROXY_SUSPEND_STATE)suspended;

	if (pNfCtx->proxyState == PS_CONNECTED)
	{
		switch (suspended)
		{
		case PSS_NONE:
			nfproxy_startTcpReadLocal(pNfCtx);
			nfproxy_startTcpReadRemote(pNfCtx);
			break;

		case PSS_SUSPEND_INBOUND:
			nfproxy_startTcpReadLocal(pNfCtx);
			break;

		case PSS_SUSPEND_OUTBOUND:
			nfproxy_startTcpReadRemote(pNfCtx);
			break;
		}
	}

	return NF_STATUS_SUCCESS;
}

NFAPI_API NF_STATUS NFAPI_NS
nf_tcpClose(ENDPOINT_ID id)
{
	AutoNfCtx pNfCtx(id);
	if (!pNfCtx)
		return NF_STATUS_FAIL;

	DbgPrint("nf_tcpClose[%llu]", id);

	if (pNfCtx->proxyState != PS_CLOSED)
	{
		nfproxy_postClose(pNfCtx);
	}

	return NF_STATUS_SUCCESS;
}

NFAPI_API NF_STATUS NFAPI_NS
nf_tcpDisableFiltering(ENDPOINT_ID id)
{
	AutoNfCtx pNfCtx(id);
	if (!pNfCtx)
		return NF_STATUS_FAIL;

	DbgPrint("nf_tcpDisableFiltering[%llu]", id);

	pNfCtx->filteringDisabled = true;

	return NF_STATUS_SUCCESS;
}

NFAPI_API NF_STATUS NFAPI_NS
nf_getTCPConnInfo(ENDPOINT_ID id, PNF_TCP_CONN_INFO pConnInfo)
{
	AutoNfCtx pNfCtx(id);
	if (!pNfCtx)
		return NF_STATUS_FAIL;

	if (pNfCtx->protocol != IPPROTO_TCP)
		return NF_STATUS_FAIL;

	*pConnInfo = pNfCtx->connInfo.tcpConnInfo;

	return NF_STATUS_SUCCESS;
}

NFAPI_API unsigned long NFAPI_NS
nf_setTCPTimeout(unsigned long timeout)
{
	return 0;
}

NFAPI_API NF_STATUS NFAPI_NS nf_udpPostSend(ENDPOINT_ID id, const unsigned char * remoteAddress, const char * buf, int len, PNF_UDP_OPTIONS options)
{
	return NF_STATUS_SUCCESS;
}

NFAPI_API NF_STATUS NFAPI_NS nf_udpPostReceive(ENDPOINT_ID id, const unsigned char * remoteAddress, const char * buf, int len, PNF_UDP_OPTIONS options)
{
	return NF_STATUS_SUCCESS;
}

NFAPI_API NF_STATUS NFAPI_NS nf_udpSetConnectionState(ENDPOINT_ID id, int suspended)
{
	return NF_STATUS_SUCCESS;
}

static void nfproxy_dispatchOperation(OV_DATA * pov)
{
	typedef void (*ovHandler)(OV_DATA * pov);

	static ovHandler ovHandlers[] = {
		nfproxy_onTcpAccepted,
		nfproxy_onTcpConnected,
		nfproxy_onClosed,
		nfproxy_onTcpLocalEvent,
		nfproxy_onTcpRemoteEvent,
		nfproxy_onTcpLocalWriteComplete,
		nfproxy_onTcpLocalReadComplete,
		nfproxy_onTcpRemoteWriteComplete,
		nfproxy_onTcpRemoteReadComplete
	};

	if (pov->type >= OVT_MAX)
	{
		DbgPrint("nfproxy_dispatchOperation[%llu] wrong type %d", 
			pov->id, pov->type);
		return;
	}

	ovHandlers[pov->type](pov);
}

static void nfproxy_dispatchOperations(PNFCTX pNfCtx, VLIST_ENTRY * opList)
{
	VLIST_ENTRY * pEntry;
	OV_DATA * pov;

	DbgPrint("nfproxy_dispatchOperations[%llu]", 
				pNfCtx->id);

	while (!vlist_isEmpty(opList))
	{
		pEntry = vlist_removeHead(opList);
		pov = vlist_record(pEntry, OV_DATA, entry);

		vlist_init(&pov->entry);

		vcs_lock(&pNfCtx->lock);

		nfproxy_dispatchOperation(pov);

		vcs_unlock(&pNfCtx->lock);

		ovData_release(pov, pNfCtx);
	}
}

static void nfproxy_addEvent(PNFCTX pNfCtx, OV_DATA * pov)
{
	bool addJob = false;

	DbgPrint("nfproxy_addEvent[%llu] type=%d", 
		pNfCtx->id, pov->type);

	if (pNfCtx->proxyState == PS_CLOSED)
	{
		ovData_release(pov, pNfCtx);
		return;
	}

	vcs_lock(&pNfCtx->lock);

	vlist_removeEntry(&pov->entry);
	vlist_insertTail(&pNfCtx->ovDataListCompleted, &pov->entry);

	pNfCtx->haveNewEvents = true;

	if (!pNfCtx->busy)
	{
		pNfCtx->busy = true;
		addJob = true;
	} else
	{
		DbgPrint("nfproxy_addEvent[%llu] type=%d, context is busy", 
			pNfCtx->id, pov->type);
	}

	vcs_unlock(&pNfCtx->lock);

	if (addJob)
	{
		vcs_lock(&g_proxyData.lock);

		vlist_insertTail(&g_proxyData.nfctxCompletedList, &pNfCtx->completedEntry);

		nfthreadpool_jobAvailable();

		vcs_unlock(&g_proxyData.lock);

		DbgPrint("nfproxy_addEvent[%llu] type=%d, job added", 
			pNfCtx->id, pov->type);
	}
}

static void nfproxy_execute()
{
	PNFCTX pNfCtx;
	VLIST_ENTRY * pEntry;
	VLIST_ENTRY opList;
	bool addJob = false;

	{
		vcs_lock(&g_proxyData.lock);

		if (vlist_isEmpty(&g_proxyData.nfctxCompletedList))
		{
			vcs_unlock(&g_proxyData.lock);
			return;
		}

		pEntry = vlist_removeHead(&g_proxyData.nfctxCompletedList);
		pNfCtx = vlist_record(pEntry, NFCTX, completedEntry);

		vlist_init(&pNfCtx->completedEntry);

		if (!vlist_isEmpty(&g_proxyData.nfctxCompletedList))
		{
			nfthreadpool_jobAvailable();
		}

		vcs_unlock(&g_proxyData.lock);
	}
	
	{
		vcs_lock(&pNfCtx->lock);

		pNfCtx->haveNewEvents = false;

		vlist_move(&opList, &pNfCtx->ovDataListCompleted);

		nfctx_addRef(pNfCtx);

		vcs_unlock(&pNfCtx->lock);
	}

	nfproxy_dispatchOperations(pNfCtx, &opList);

	{
		vcs_lock(&pNfCtx->lock);

		if (pNfCtx->haveNewEvents && 
			pNfCtx->proxyState != PS_CLOSED)
		{
			addJob = true;
		} else
		{
			pNfCtx->busy = false;
		}

		vcs_unlock(&pNfCtx->lock);

		if (addJob)
		{
			vcs_lock(&g_proxyData.lock);

			vlist_insertTail(&g_proxyData.nfctxCompletedList, &pNfCtx->completedEntry);
	
			nfthreadpool_jobAvailable();

			DbgPrint("nfproxy_execute[%llu] job added", 
				pNfCtx->id);

			vcs_unlock(&g_proxyData.lock);
		}
	}

	nfctx_release(pNfCtx);
}

static void nfproxy_onAccept()
{
	int s;
	int err;
	sockaddr_in6 addr;
	socklen_t addrSize;

	DbgPrint("nfproxy_onAccept");

	addrSize = sizeof(addr);
 	s = accept(g_proxyData.listenSocket, (struct sockaddr *)&addr, &addrSize);	
	if (s == INVALID_SOCKET)
	{
		DbgPrint("nfproxy_onAccept accept error %d", errno);
		return;
	}

	int optVal = 1;
	if (setsockopt(s, IPPROTO_TCP, TCP_NODELAY, &optVal, sizeof(optVal)) == -1)
	{
		DbgPrint("nfproxy_startListen socket %d, TCP_NODELAY error %d", s, errno);
	}

	PNFCTX pNfCtx = nfproxy_addNewNfCtx();
	if (!pNfCtx)
	{
		DbgPrint("nfproxy_onAccept nfproxy_addNewNfCtx error %d", errno);
		close(s);
		return;
	}

	pNfCtx->localSocket = s;

	pNfCtx->connInfoReal.tcpConnInfo.direction = NF_D_OUT;

	memcpy(pNfCtx->connInfoReal.tcpConnInfo.localAddress, &addr, addrSize);

	addrSize = sizeof(addr);
	err = getsockname(s, (struct sockaddr *)pNfCtx->connInfoReal.tcpConnInfo.remoteAddress, &addrSize);
	if (err != 0)
	{
		DbgPrint("nfproxy_onAccept getsockname error %d", errno);
	}

	if (nfproxy_isLocalAddress((sockaddr*)pNfCtx->connInfoReal.tcpConnInfo.remoteAddress))
	{
		unsigned short remotePort = ntohs(((sockaddr_in6*)pNfCtx->connInfoReal.tcpConnInfo.remoteAddress)->sin6_port);

		if (g_proxyData.listenPort == remotePort)
		{
			DbgPrint("nfproxy_onAccept inbound connection to local proxy address, closing");
			nfctx_free(pNfCtx);
			return;
		}

		if (memcmp(&((sockaddr_in6*)pNfCtx->connInfoReal.tcpConnInfo.remoteAddress)->sin6_addr,
				&((sockaddr_in6*)pNfCtx->connInfoReal.tcpConnInfo.localAddress)->sin6_addr,
				16) != 0)
		{
			DbgPrint("nfproxy_onAccept inbound connection");
			pNfCtx->connInfoReal.tcpConnInfo.direction = NF_D_IN;
		}
	}

	dbgPrintAddress(pNfCtx->connInfoReal.tcpConnInfo.localAddress, "local real", pNfCtx->id);
	dbgPrintAddress(pNfCtx->connInfoReal.tcpConnInfo.remoteAddress, "remote real", pNfCtx->id);

	memcpy(&pNfCtx->connInfo, &pNfCtx->connInfoReal, sizeof(pNfCtx->connInfo));

	if (nfproxy_copyMappedAddress(
		(sockaddr*)pNfCtx->connInfo.tcpConnInfo.localAddress, 
		(sockaddr*)pNfCtx->connInfoReal.tcpConnInfo.localAddress))
	{
		pNfCtx->connInfo.tcpConnInfo.ip_family = AF_INET;
	} else
	{
		pNfCtx->connInfo.tcpConnInfo.ip_family = AF_INET6;
	}

	nfproxy_copyMappedAddress(
		(sockaddr*)pNfCtx->connInfo.tcpConnInfo.remoteAddress, 
		(sockaddr*)pNfCtx->connInfoReal.tcpConnInfo.remoteAddress);

	dbgPrintAddress(pNfCtx->connInfo.tcpConnInfo.localAddress, "local", pNfCtx->id);
	dbgPrintAddress(pNfCtx->connInfo.tcpConnInfo.remoteAddress, "remote", pNfCtx->id);

	OV_DATA * pov = ovData_alloc(OVT_TCP_ACCEPTED, pNfCtx);
	if (!pov)
	{
		DbgPrint("nfproxy_onAccept ovData_alloc error %d", errno);
		nfctx_free(pNfCtx);
		return;
	}

	nfproxy_addEvent(pNfCtx, pov);
}

static void nfproxy_processTimeouts()
{
	VLIST_ENTRY * pEntry;
	PNFCTX pNfCtx;
	time_t curTime = time(NULL);
	static time_t prevTime = 0;
	const int maxEndpoints = 10;
	ENDPOINT_ID endpoints[maxEndpoints] = {};
	int nEndpoints = 0;
	int i;

	if ((curTime - prevTime) < 1)
		return;

	prevTime = curTime;

	vcs_lock(&g_proxyData.lock);

	vlist_forEach(&g_proxyData.nfctxList, pEntry)
	{
		pNfCtx = vlist_record(pEntry, NFCTX, entry);

		if (pNfCtx->ioTimeout == 0)
			continue;

		if ((pNfCtx->proxyState != PS_CLOSED) &&
			(curTime - pNfCtx->ioTime) > pNfCtx->ioTimeout)
		{
			pNfCtx->ioTimeout = 0;

			endpoints[nEndpoints] = pNfCtx->id;
			nEndpoints++;

			if (nEndpoints >= maxEndpoints)
				break;
		}
	}

	vcs_unlock(&g_proxyData.lock);

	for (i=0; i<nEndpoints; i++)
	{
		AutoNfCtx p(endpoints[i]);
		if (!p)
			continue;

		nfproxy_postClose(p);
	}
}

static void nfproxy_onEvent(uint64_t data, int events)
{
	if (data == 0 && events == 0)
	{
		// epoll_wait timeout occurred
		nfproxy_processTimeouts();
		return;
	}

	if (data == 0)
	{
		// the listening socket uses zero id, and events must be EPOLLIN
		nfproxy_onAccept();
		return;
	}

	// Local socket uses odd id, equal to the endpoint id of NFCTX.
	// Remote socket uses even id, equal to endpoint id + 1.

	int ovType = ((data & 1) != 0)? OVT_TCP_LOCAL_EVENT : OVT_TCP_REMOTE_EVENT;
	uint64_t id = ((data & 1) != 0)? data : data - 1;

	AutoNfCtx pNfCtx(id);
	if (!pNfCtx)
	{
		DbgPrint("nfproxy_onEvent[%llu] context not found", id);
		return;
	}

	DbgPrint("nfproxy_onEvent[%llu] isLocal=%d, events=0x%x", 
		pNfCtx->id, (data & 1), events);

	pNfCtx->ioTime = time(NULL);

	OV_DATA * pov = ovData_alloc(ovType, pNfCtx);
	if (!pov)
	{
		DbgPrint("nfproxy_onEvent ovData_alloc error %d", errno);
		return;
	}

	pov->transferred = (size_t)events;

	nfproxy_addEvent(pNfCtx, pov);

	nfproxy_processTimeouts();
}

static int nfproxy_startListen(unsigned short port)
{
	int s;
	int err;

	DbgPrint("nfproxy_startListen port %d", port);

	// Using a single dual stack TCP IPv6 socket for accepting both IPv4 and IPv6 connections
	
	s = socket(AF_INET6, SOCK_STREAM | SOCK_NONBLOCK, 0);
	if (s == INVALID_SOCKET)
		return INVALID_SOCKET;

	for (;;)
	{
		int mode = 0;
		if (setsockopt(s, IPPROTO_IPV6, IPV6_V6ONLY, (char*)&mode, sizeof(mode)) != 0)
		{
			DbgPrint("nfproxy_startListen setsockopt error %d", errno);
			break;
		}

		int optVal = 1;
		if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR, &optVal, sizeof(optVal)) == -1)
		{
			DbgPrint("nfproxy_startListen SO_REUSEADDR error %d", errno);
			break;
		}

		optVal = 1;
	 	if (setsockopt(s, IPPROTO_TCP, TCP_NODELAY, &optVal, sizeof(optVal)) == -1)
		{
			DbgPrint("nfproxy_startListen socket %d, TCP_NODELAY error %d", s, errno);
        	break;
	    }

		if (setsockopt(s, SOL_IP, IP_TRANSPARENT, &optVal, sizeof(optVal)) == -1)
		{
			DbgPrint("nfproxy_startListen IP_TRANSPARENT error %d", errno);
			break;
		}

		sockaddr_in6 addr;
		memset(&addr, 0, sizeof(addr));
		addr.sin6_family = AF_INET6;
		addr.sin6_port = htons(port);

		err = bind(s, (sockaddr*)&addr, sizeof(addr));
		if (err != 0)
		{
			DbgPrint("nfproxy_startListen bind error %d", errno);
			break;
		}

		err = listen(s, LISTEN_BACKLOG);
		if (err != 0)
		{
			DbgPrint("nfproxy_startListen listen error %d", errno);
			break;
		}

		if (!nfiocp_registerSocket(s, 0, EPOLLIN))
		{
			DbgPrint("nfproxy_startListen nfiocp_registerSocket error %d", errno);
			break;
		}

		return s;
	}

	close(s);

	return INVALID_SOCKET;
}

NFAPI_API unsigned short NFAPI_NS
nf_getProxyPort()
{
	return g_proxyData.listenPort;
}

NFAPI_API unsigned long NFAPI_NS
nf_getConnCount()
{
	return g_proxyData.nfctxListSize;
}

NFAPI_API NF_STATUS NFAPI_NS 
nf_getUid(ENDPOINT_ID id, unsigned int * pUid)
{
	AutoNfCtx pNfCtx(id);
	if (!pNfCtx)
		return NF_STATUS_FAIL;

	*pUid = pNfCtx->uid;

	return NF_STATUS_SUCCESS;
}

NFAPI_API void NFAPI_NS
nf_setOptions(int nThreads, int port)
{
	g_nThreads = nThreads;
	
	if (g_nThreads <= 0)
	{
		g_nThreads = get_nprocs();
		if (g_nThreads <= 0)
			g_nThreads = 1;
	} else
	if (g_nThreads > 1024)
	{
		g_nThreads = 1024;
	}

	g_listenPort = port;

	if (g_listenPort == 0 || g_listenPort > 65535)
		g_listenPort = 8888;
}

NFAPI_API NF_STATUS NFAPI_NS
nf_init(const char * driverName, NFAPI_NS NF_EventHandler * pEventHandler)
{
	bool listenStarted = false;

	if (g_initialized)
		return NF_STATUS_SUCCESS;

	nfrules_setDriverName(driverName);

#ifdef _RELEASE_LOG
	dbglog_init("nfproxylog.txt");
	dbglog_enable(TRUE);
#endif

	DbgPrint("nf_init (%d) threads", g_nThreads);

	if (!pEventHandler)
	{
		DbgPrint("nf_init !pEventHandler");
		return NF_STATUS_FAIL;
	}

	memset(&g_proxyData, 0, sizeof(g_proxyData));

	g_proxyData.listenSocket = INVALID_SOCKET;
	g_proxyData.listenPort = g_listenPort;

	g_proxyData.pEventHandler = pEventHandler;

	vlist_init(&g_proxyData.nfctxList);
	vlist_init(&g_proxyData.nfctxCompletedList);

	if (!vcs_init(&g_proxyData.lock))
	{
		return NF_STATUS_FAIL;
	}

	g_proxyData.nextId = 1;

	g_initialized = true;
	
	for (;;)
	{
		g_proxyData.phtNfCtx = hash_table_new(HASHTABLE_SIZE);
		if (!g_proxyData.phtNfCtx)
			break;

		THREAD_JOB_SOURCE tjs = {
			nfproxy_execute,
			nfproxy_threadStarted,
			nfproxy_threadStopped
		};

		if (!nfthreadpool_init(g_nThreads, &tjs))
		{
			DbgPrint("nf_init nfthreadpool_init failed");
			break;
		}

		if (!nfiocp_init(nfproxy_onEvent))
		{
			DbgPrint("nf_init nfiocp_init failed");
			break;
		}

		for (int i=0; i<100; i++)
		{
			g_proxyData.listenSocket = nfproxy_startListen(g_proxyData.listenPort);
			if (g_proxyData.listenSocket == INVALID_SOCKET)
			{
				DbgPrint("nf_init nfproxy_startListen %d failed", g_proxyData.listenSocket);
				g_proxyData.listenPort++;				
				continue;
			}
			listenStarted = true;
			break;
		}

		if (!listenStarted)
			break;		

		nf_addRouteRules();

		return NF_STATUS_SUCCESS;
	}

	DbgPrint("nf_init initialization failed");

	nf_free();

	return NF_STATUS_FAIL;
}

NFAPI_API void NFAPI_NS
nf_free()
{
	VLIST_ENTRY * pEntry;
	PNFCTX pNfCtx;

	if (!g_initialized)
		return;

	DbgPrint("nf_free");

	g_initialized = false;

	nf_deleteRouteRules();

	nf_deleteRules();

	nfthreadpool_free();
	DbgPrint("nf_free nfthreadpool_free finished");

	nfiocp_free();
	DbgPrint("nf_free nfiocp_free finished");

	if (g_proxyData.listenSocket != INVALID_SOCKET)
	{
		close(g_proxyData.listenSocket);
		g_proxyData.listenSocket = INVALID_SOCKET;
	}

	while (!vlist_isEmpty(&g_proxyData.nfctxList))
	{
		pEntry = vlist_removeHead(&g_proxyData.nfctxList);
		pNfCtx = vlist_record(pEntry, NFCTX, entry);

		vlist_init(&pNfCtx->entry);

		nfctx_free(pNfCtx);
	}

	if (g_proxyData.phtNfCtx)
	{
		hash_table_free(g_proxyData.phtNfCtx);
		g_proxyData.phtNfCtx = NULL;
	}

	g_proxyData.pEventHandler = NULL;

	vcs_free(&g_proxyData.lock);

	DbgPrint("nf_free completed");

#ifdef _RELEASE_LOG
	dbglog_free();
#endif
}

