//
// 	NetFilterSDK 
// 	Copyright (C) Vitaly Sidorov
//	All rights reserved.
//
//	This file is a part of the NetFilter SDK.
//	The code and information is provided "as-is" without
//	warranty of any kind, either expressed or implied.
//

#include "vdefs.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/sysinfo.h>
#include "vthread.h"
#include "vsync.h"
#include "dbglog.h"
#include "nfthreadpool.h"

using namespace mainNS;

enum ThreadEventState
{
	TES_NONE = 0,
	TES_JOB_AVAILABLE = 1,
	TES_STOP = 2
};

struct THREADPOOL_DATA
{
	THREAD_JOB_SOURCE jobSource;
	vthread_t *		threadsArray;
	unsigned int 	threadCount;
	vsyncEvent_t	threadEvent;
};

static bool g_initialized = false;
static THREADPOOL_DATA g_tpData;

static VTHREAD_FUNC nfthreadpool_workerThread(void*)
{
	int flags;

	g_tpData.jobSource.threadStarted();

	for (;;)
	{
		vsyncEvent_wait(g_tpData.threadEvent, &flags);

		if (!g_initialized)
			break;

		g_tpData.jobSource.execute();
	}

	g_tpData.jobSource.threadStopped();

	return 0;
}

void mainNS::nfthreadpool_jobAvailable()
{
	if (!g_initialized)
		return;

	vsyncEvent_fire(g_tpData.threadEvent, TES_JOB_AVAILABLE);
}

bool mainNS::nfthreadpool_init(int threadCount, THREAD_JOB_SOURCE * pJobSource)
{
	DbgPrint("nfthreadpool_init");

	if (g_initialized)
		return true;

	g_initialized = true;

	memset(&g_tpData, 0, sizeof(g_tpData));

	for (;;)
	{
		g_tpData.jobSource = *pJobSource;

		if (threadCount <= 0)
		{
			threadCount = get_nprocs();
			if (threadCount == 0)
			{
				threadCount = 1;
			}
		}

		g_tpData.threadEvent = vsyncEvent_create();
		if (g_tpData.threadEvent == NULL)
			break;

		g_tpData.threadsArray = (vthread_t*)malloc(sizeof(vthread_t) * threadCount);
		if (g_tpData.threadsArray == NULL)
			break;

		memset(g_tpData.threadsArray, 0, sizeof(vthread_t) * threadCount);

		g_tpData.threadCount = threadCount;

		for (int i=0; i<threadCount; i++)
		{
			g_tpData.threadsArray[i] = vthread_create(nfthreadpool_workerThread, NULL);
			if (g_tpData.threadsArray[i] == 0)
			{
				nfthreadpool_free();
				return false;
			}
		}

		return true;
	}

	DbgPrint("nfthreadpool_init failed");

	nfthreadpool_free();

	return false;
}

void mainNS::nfthreadpool_free()
{
	DbgPrint("nfthreadpool_free");

	if (!g_initialized)
		return;

	g_initialized = false;

	if (g_tpData.threadsArray != NULL)
	{
		vsyncEvent_fireAll(g_tpData.threadEvent, TES_STOP);

		for (unsigned int i = 0; i < g_tpData.threadCount; i++)
		{
			if (g_tpData.threadsArray[i] != 0)
			{
				vthread_wait(g_tpData.threadsArray[i]);
				g_tpData.threadsArray[i] = 0;
			}
		}

		::free(g_tpData.threadsArray);
		g_tpData.threadsArray = NULL;
	}

	if (g_tpData.threadEvent != NULL)
	{
		vsyncEvent_free(g_tpData.threadEvent);
		g_tpData.threadEvent = NULL;
	}
}
