//
// 	NetFilterSDK 
// 	Copyright (C) Vitaly Sidorov
//	All rights reserved.
//
//	This file is a part of the NetFilter SDK.
//	The code and information is provided "as-is" without
//	warranty of any kind, either expressed or implied.
//

#include "vdefs.h"
#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>
#include <dlfcn.h>
#include <fcntl.h>
#include <glob.h>
#include <syslog.h>
#include <pwd.h> 
#include <grp.h> 
#include <sys/resource.h>
#include <dirent.h>
#include <errno.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <string.h>
#include <sys/types.h>
#include <ifaddrs.h>

#include "dbglog.h"
#include "nfapi_linux.h"
#include "nfutil.h"

using namespace mainNS;

NFAPI_API NF_STATUS NFAPI_NS 
nf_getUserName(unsigned int uid, char * buf, int len)
{
	struct passwd * pwd;
	NF_STATUS res = NF_STATUS_FAIL;

	pwd = getpwuid(uid);
	if (pwd)
	{
		snprintf(buf, len, "%s", pwd->pw_name);
		res = NF_STATUS_SUCCESS;
	}

	return res;
}

NFAPI_API NF_STATUS NFAPI_NS 
nf_getProcessName(unsigned int pid, char * buf, int len)
{
    char path[PATH_MAX];
	NF_STATUS res = NF_STATUS_FAIL;
	size_t linkLen;

	snprintf(path, sizeof(path), "/proc/%u/exe", pid);

    linkLen = readlink(path, buf, len-1);
	if (linkLen != -1)
	{
		buf[linkLen] = '\0';

		res = NF_STATUS_SUCCESS;
	}

	return res;
}

NFAPI_API NF_STATUS NFAPI_NS 
nf_requireFileLimit(int file_limit)
{
	struct rlimit lim;

	if (getrlimit(RLIMIT_NOFILE, &lim) != 0)
	{
		return NF_STATUS_FAIL;
	}

	if (lim.rlim_cur >= file_limit)
		return NF_STATUS_SUCCESS;

	lim.rlim_cur = lim.rlim_max = file_limit;

	if (setrlimit(RLIMIT_NOFILE, &lim) != 0)
	{
		return NF_STATUS_FAIL;
	}

	if (getrlimit(RLIMIT_NOFILE, &lim) != 0)
	{
		return NF_STATUS_FAIL;
	}

	return NF_STATUS_SUCCESS;
}

NFAPI_API NF_STATUS NFAPI_NS 
nf_requireStackLimit(int stack_limit)
{
	struct rlimit lim;

	if (getrlimit(RLIMIT_STACK, &lim) != 0)
	{
		return NF_STATUS_FAIL;
	}

	if (lim.rlim_cur >= stack_limit)
		return NF_STATUS_SUCCESS;

	lim.rlim_cur = lim.rlim_max = stack_limit;

	if (setrlimit(RLIMIT_STACK, &lim) != 0)
	{
		return NF_STATUS_FAIL;
	}

	return NF_STATUS_SUCCESS;
}

static bool nfutil_findProcessId(TCP_PROC_INFO * pInfo)
{
	pid_t pid = 0;
	char path[PATH_MAX] = {};
	int pathLen;
	char link_src[PATH_MAX] = {};
	struct stat statbuf;
	char* endptr;
	bool pidFound = false;

	DbgPrint("nfutil_findProcessId inode=%d uid=%d", pInfo->iNode, pInfo->uid);

	DIR *dir = opendir("/proc");
	if (dir == NULL) 
	{
		return false;
	}
	
	struct dirent* de;
	while ((de = readdir(dir))) 
	{
		if (!strcmp(de->d_name, ".") || !strcmp(de->d_name, ".."))
			continue;

		pid = strtol(de->d_name, &endptr, 10);

		if (*endptr != '\0')
			continue;

		snprintf(path, sizeof(path)-1, "/proc/%d/fd", pid);

		DIR *dirFd = opendir(path);
		if (dirFd == NULL) 
		{
			DbgPrint("nfutil_findProcessId opendir failed for %s", path);
			continue;
		}

		_snprintf(link_src, sizeof(link_src)-1, "%s/", path);
		pathLen = strlen(path)+1;

		struct dirent* deFd;
		while ((deFd = readdir(dirFd))) 
		{
			if (!strcmp(deFd->d_name, ".") || !strcmp(deFd->d_name, ".."))
				continue;

			link_src[pathLen] = '\0';

			strncat(link_src, deFd->d_name, sizeof(link_src)-1);

			if (stat(link_src, &statbuf) == 0)
			{
				if (S_ISSOCK(statbuf.st_mode))
				{
					if (pInfo->iNode == statbuf.st_ino)
					{
						pInfo->pid = pid;
						pidFound = true;
						break;
					}
				}
			}
		}

		closedir(dirFd);

		if (pidFound)
			break;
	}

	closedir(dir);

	return pidFound;
}

static bool nfutil_getTCPProcInfoProto(int ip_family, unsigned short localPortToFind, TCP_PROC_INFO * pInfo)
{
	FILE * f;
	unsigned long rxq, txq, time_len, retr, inode;
	int num, local_port, rem_port, d, state, uid, timer_run, timeout;
	char rem_addr[128], local_addr[128];
	char buffer[8192];	
	bool res = false;

	DbgPrint("nfutil_getTCPProcInfo port %x", localPortToFind);

	if (ip_family == AF_INET)
	{
		f = fopen("/proc/net/tcp", "r");
	} else
	{
		f = fopen("/proc/net/tcp6", "r");
	}
	if (!f)
	{
		DbgPrint("nfutil_getTCPProcInfo unable to open /proc/net/tcp[6]");
		return false;
	}

	if (!fgets(buffer, sizeof(buffer), f))
		return false;
	
	do {						
		if (!fgets(buffer, sizeof(buffer), f))
			break;

		num = sscanf(buffer,
			"%d: %64[0-9A-Fa-f]:%X %64[0-9A-Fa-f]:%X %X %lX:%lX %X:%lX %lX %d %d %lu %*s\n",
				&d, local_addr, &local_port, rem_addr, &rem_port, &state,
				&txq, &rxq, &timer_run, &time_len, &retr, &uid, &timeout, &inode);

		if (num < 11) 
		{
			DbgPrint("nfutil_getTCPProcInfo sscanf failed %s", buffer);
			break;
		}

//		DbgPrint("nfutil_getTCPProcInfo %s", buffer);

		if (localPortToFind == local_port)
		{
			pInfo->iNode = inode;
			pInfo->uid = uid;

			DbgPrint("nfutil_getTCPProcInfo uid=%d", pInfo->uid);

			res = true;

			if (nfutil_findProcessId(pInfo))
			{
				DbgPrint("nfutil_getTCPProcInfo pid=%d", pInfo->pid);
			}
			break;
		}

	} while (!feof(f));		


	fclose(f);

	return res;
}

bool mainNS::nfutil_getTCPProcInfo(unsigned short localPortToFind, TCP_PROC_INFO * pInfo)
{
	memset(pInfo, 0, sizeof(TCP_PROC_INFO));

	if (nfutil_getTCPProcInfoProto(AF_INET, localPortToFind, pInfo))
	{
		return true;
	}

	if (nfutil_getTCPProcInfoProto(AF_INET6, localPortToFind, pInfo))
	{
		return true;
	}

	return false;
}
