//
// 	NetFilterSDK 
// 	Copyright (C) Vitaly Sidorov
//	All rights reserved.
//
//	This file is a part of the NetFilter SDK.
//	The code and information is provided "as-is" without
//	warranty of any kind, either expressed or implied.
//

#include "vdefs.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "vbuffer.h"

using namespace mainNS;

typedef struct _VBUFFER
{
	VLIST_ENTRY		listEntry;
	char*			buffer;
	unsigned long	size;
	unsigned long	capacity;
	unsigned long	limit;
	unsigned long	reservePercent;
} VBUFFER, * PVBUFFER;

vbuffer_t mainNS::vbuffer_create()
{
	PVBUFFER vbuf;

    vbuf = vmalloc(VBUFFER);
    if (!vbuf)
        return NULL;

    memset(vbuf, 0, sizeof(VBUFFER));
    vbuf->limit = 8 * 1024 * 1024;
	vbuf->reservePercent = 0;

    return (vbuffer_t)vbuf;
}

void mainNS::vbuffer_free(vbuffer_t b)
{
	PVBUFFER vbuf = (PVBUFFER)b;

    if (!vbuf)
        return;

    if (vbuf->buffer)
    {
        free(vbuf->buffer);
    }

    free(vbuf);
}

BOOL mainNS::vbuffer_copy(vbuffer_t dst, vbuffer_t src)
{
	PVBUFFER vbuf = (PVBUFFER)dst;

    if (!vbuf)
        return FALSE;

	vbuf->size = 0;
	vbuf->capacity = 0;
	vbuf->limit = ((PVBUFFER)src)->limit;
	vbuf->reservePercent = ((PVBUFFER)src)->reservePercent;

	if (vbuf->buffer)
	{
		free(vbuf->buffer);
		vbuf->buffer = NULL;
	}

	if (!vbuffer_append(dst, vbuffer_buffer(src), vbuffer_getSize(src), TRUE))
		return FALSE;

	return TRUE;
}

BOOL mainNS::vbuffer_append(vbuffer_t b, const char* buf, unsigned long len, bool copy)
{
	unsigned long	uRequired;
	char* p;
	PVBUFFER vbuf = (PVBUFFER)b;

	if ((!buf && copy) || len == 0)
		return false;

	uRequired = vbuf->size + len + 2;

	if (uRequired > vbuf->capacity)
	{
		uRequired = (uRequired * (100 + vbuf->reservePercent)) / 100;

		if (uRequired >= vbuf->limit)
		{
			return false;
		}

		if (vbuf->buffer)
		{
			p = (char*)realloc(vbuf->buffer, uRequired);
			if (p == NULL)
			{
				return false;
			}
		}
		else
		{
			p = (char*)malloc(uRequired);
			if (p == NULL)
			{
				return false;
			}
		}

		vbuf->capacity = uRequired;

		vbuf->buffer = p;
	}

	if (copy)
	{
		memcpy(vbuf->buffer + vbuf->size, buf, (size_t)len);
	}

	vbuf->size += len;

	vbuf->buffer[vbuf->size] = '\0';
	vbuf->buffer[vbuf->size+1] = '\0';

	return true;
}

BOOL mainNS::vbuffer_appendByte(vbuffer_t b, char c)
{
	return vbuffer_append(b, &c, 1, TRUE);
}

BOOL mainNS::vbuffer_appendString(vbuffer_t b, const char* str)
{
	return vbuffer_append(b, str, (unsigned long)strlen(str), TRUE);
}

#ifdef WIN32
BOOL mainNS::vbuffer_appendWString(vbuffer_t b, const wchar_t* str)
{
	return vbuffer_append(b, (char*)str, (unsigned long)wcslen(str)*sizeof(wchar_t), TRUE);
}
#endif

unsigned long mainNS::vbuffer_getSize(vbuffer_t b)
{
	return ((PVBUFFER)b)->size;
}

void mainNS::vbuffer_reset(vbuffer_t b)
{
	PVBUFFER vbuf = (PVBUFFER)b;

	vbuf->size = 0;
	vbuf->capacity = 0;

	if (vbuf->buffer)
	{
		free(vbuf->buffer);
		vbuf->buffer = NULL;
	}
}

void mainNS::vbuffer_clear(vbuffer_t b)
{
	PVBUFFER vbuf = (PVBUFFER)b;

	vbuf->size = 0;

	if (vbuf->buffer)
	{
		vbuf->buffer[0] = 0;
	}
}

char* mainNS::vbuffer_buffer(vbuffer_t b)
{
	return ((PVBUFFER)b)->buffer;
}

void mainNS::vbuffer_setLimit(vbuffer_t b, unsigned long limit)
{
	((PVBUFFER)b)->limit = limit;
}

void mainNS::vbuffer_setReservePercent(vbuffer_t b, unsigned long reservePercent)
{
	((PVBUFFER)b)->reservePercent = reservePercent;
}

PVLIST_ENTRY mainNS::vbuffer_getListEntry(vbuffer_t b)
{
	return &((PVBUFFER)b)->listEntry;
}

vbuffer_t mainNS::vbuffer_fromListEntry(PVLIST_ENTRY listEntry)
{
	return (vbuffer_t)vlist_record(listEntry, VBUFFER, listEntry);
}

