//
// 	NetFilterSDK 
// 	Copyright (C) Vitaly Sidorov
//	All rights reserved.
//
//	This file is a part of the NetFilter SDK.
//	The code and information is provided "as-is" without
//	warranty of any kind, either expressed or implied.
//

#include "vdefs.h"
#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>
#include <string.h>
#include "vsync.h"

using namespace mainNS;

#ifdef WIN32

vcs_t mainNS::vcs_create()
{
    VCS * pcs;

    pcs = vmalloc(VCS);
    if (!pcs)
        return NULL;

	InitializeCriticalSection(&pcs->cs);
	
	pcs->flags = VF_DYNAMIC;

    return pcs;
}

void mainNS::vcs_init(vcs_t b)
{
    InitializeCriticalSection((CRITICAL_SECTION*)b);
	
	((VCS*)b)->flags = VF_STATIC;
}

void mainNS::vcs_lock(vcs_t b)
{
	EnterCriticalSection((CRITICAL_SECTION*)b);
}

void mainNS::vcs_unlock(vcs_t b)
{
    LeaveCriticalSection((CRITICAL_SECTION*)b);
}

void mainNS::vcs_free(vcs_t b)
{
    DeleteCriticalSection((CRITICAL_SECTION*)b);

	if (((VCS*)b)->flags == VF_DYNAMIC)
		free(b);
}

#else

vcs_t mainNS::vcs_create()
{
    VCS * pcs;

    pcs = vmalloc(VCS);
    if (!pcs)
        return NULL;

    if (!vcs_init(pcs))
    {
        free(pcs);
        return NULL;
    }

	pcs->flags = VF_DYNAMIC;

    return pcs;
}

bool mainNS::vcs_init(vcs_t b)
{
    VCS * pcs = (VCS*)b;
    pthread_mutexattr_t attr;
    bool result = false;

    for (;;)
    {
        if (pthread_mutexattr_init(&attr) != 0)
        {
            break;
        }

        if (pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_RECURSIVE) != 0)
        {
            break;
        }

        if (pthread_mutex_init(&pcs->cs, &attr) != 0)
        {
            break;
        }

        pthread_mutexattr_destroy(&attr);

        result = true;

        break;
    }

	pcs->flags = VF_STATIC;

    return result;
}

void mainNS::vcs_lock(vcs_t b)
{
    pthread_mutex_lock(&((VCS*)b)->cs);
}

void mainNS::vcs_unlock(vcs_t b)
{
    pthread_mutex_unlock(&((VCS*)b)->cs);
}

void mainNS::vcs_free(vcs_t b)
{
	pthread_mutex_destroy(&((VCS*)b)->cs);

	if (((VCS*)b)->flags == VF_DYNAMIC)
		free(b);
}

#endif

vsyncEvent_t mainNS::vsyncEvent_create()
{
    PSYNC_EVENT p = vmalloc(SYNC_EVENT);
    if (!p)
        return NULL;

    memset(p, 0, sizeof(SYNC_EVENT));

    for (;;)
    {
#ifdef WIN32
        p->event = CreateEvent(NULL, FALSE, FALSE, NULL);
        if (!p->event)
            break;
#else
        if (pthread_cond_init(&p->event, NULL) != 0)
        {
            break;
        }
#endif
        p->cs = vcs_create();
        if (!p->cs)
            break;

        return p;
    }

    vsyncEvent_free(p);

    return NULL;
}

BOOL mainNS::vsyncEvent_fire(vsyncEvent_t b, int flags)
{
    PSYNC_EVENT p = (PSYNC_EVENT)b;

    vcs_lock(p->cs);
    p->flags |= flags;
    vcs_unlock(p->cs);

#ifdef WIN32
    return SetEvent(p->event) != 0;
#else
	return (pthread_cond_signal(&p->event) == 0);
#endif
}

BOOL mainNS::vsyncEvent_fireAll(vsyncEvent_t b, int flags)
{
    PSYNC_EVENT p = (PSYNC_EVENT)b;

    vcs_lock(p->cs);
    p->flags |= flags;
    vcs_unlock(p->cs);

#ifdef WIN32
    return SetEvent(p->event) != 0;
#else
	return (pthread_cond_broadcast(&p->event) == 0);
#endif
}

BOOL mainNS::vsyncEvent_wait(vsyncEvent_t b, int* pFlags)
{
    PSYNC_EVENT p = (PSYNC_EVENT)b;
    AutoVCS lock(p->cs);

    if (p->flags != 0)
    {
        if (pFlags)
        {
            *pFlags = p->flags;
        }
        p->flags = 0;
        return TRUE;
    }

#ifdef WIN32    
    if (WaitForSingleObject(p->event, INFINITE) == WAIT_OBJECT_0)
#else
	if (pthread_cond_wait(&p->event, &((VCS*)p->cs)->cs) == 0)
#endif
    {
        if (pFlags)
        {
            *pFlags = p->flags;
        }
        p->flags = 0;

        return TRUE;
    }

    return FALSE;
}

BOOL mainNS::vsyncEvent_wait(vsyncEvent_t b, int* pFlags, unsigned int timeout)
{
    PSYNC_EVENT p = (PSYNC_EVENT)b;
    AutoVCS lock(p->cs);

    if (p->flags != 0)
    {
        if (pFlags)
        {
            *pFlags = p->flags;
        }
        p->flags = 0;
        return TRUE;
    }

#ifdef WIN32    
    if (WaitForSingleObject(p->event, timeout*1000) == WAIT_OBJECT_0)
#else
	struct timespec absTime;
    clock_gettime(CLOCK_REALTIME, &absTime);
    absTime.tv_sec += timeout;

	if (pthread_cond_timedwait(&p->event, &((VCS*)p->cs)->cs, &absTime) == 0)
#endif
    {
        if (pFlags)
        {
            *pFlags = p->flags;
        }
        p->flags = 0;

        return TRUE;
    }

    return FALSE;
}

BOOL mainNS::vsyncEvent_isFlagFired(vsyncEvent_t b, int flag)
{
    BOOL result;
    PSYNC_EVENT p = (PSYNC_EVENT)b;

    vcs_lock(p->cs);
    result = (p->flags & flag) != 0;
    vcs_unlock(p->cs);

    return result;
}

void mainNS::vsyncEvent_free(vsyncEvent_t b)
{
    PSYNC_EVENT p = (PSYNC_EVENT)b;
    if (!p)
        return;

#ifdef WIN32
    if (p->event != 0)
    {
        CloseHandle(p->event);
    }
#else
	pthread_cond_destroy(&p->event);
#endif        

    if (p->cs != 0)
    {
        vcs_free(p->cs);
    }

    free(p);
}
