/**
*	The sample filters HTTP connections in pass-through mode,
*	and prints the information about all called events to standard console output.
**/

#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <string>

#include "nfapi_linux.h"


using namespace nfapi;

// Change this string after renaming and registering the driver under different name
#define NFEXT_BUNDLEID	"nfext"
#define NFDRIVER_NAME NFEXT_BUNDLEID

//
//	API events handler
//
class EventHandler : public NF_EventHandler
{
	virtual void threadStart()
	{
		printf("threadStart\n");
		fflush(stdout);

		// Initialize thread specific stuff
	}

	virtual void threadEnd()
	{
		printf("threadEnd\n");

		// Uninitialize thread specific stuff
	}
	
	//
	// TCP events
	//

	virtual void tcpConnectRequest(ENDPOINT_ID id, PNF_TCP_CONN_INFO pConnInfo)
	{
		printf("tcpConnectRequest id=%llu, processId=%u\n", id, pConnInfo->processId);
	}

	virtual void tcpConnected(ENDPOINT_ID id, PNF_TCP_CONN_INFO pConnInfo)
	{
		printf("tcpConnected id=%llu processId=%d\n", id, pConnInfo->processId);

		char path[MAX_PATH];
		int ret;
		
		if (pConnInfo->ip_family == AF_INET)
		{
			std::string sLocalAddr = inet_ntoa(((sockaddr_in*)pConnInfo->localAddress)->sin_addr);
			std::string sRemoteAddr = inet_ntoa(((sockaddr_in*)pConnInfo->remoteAddress)->sin_addr);

			printf("local=%s:%d remote=%s:%d ", 
							sLocalAddr.c_str(), ntohs(((sockaddr_in*)pConnInfo->localAddress)->sin_port),
							sRemoteAddr.c_str(), ntohs(((sockaddr_in*)pConnInfo->remoteAddress)->sin_port));
		} else
		{
			char sAddr[100];
			std::string sLocalAddr;
			std::string sRemoteAddr;

			if (inet_ntop(AF_INET6, &((sockaddr_in6*)pConnInfo->localAddress)->sin6_addr, sAddr, sizeof(sAddr)))
			{
				sLocalAddr = sAddr;
			}
			if (inet_ntop(AF_INET6, &((sockaddr_in6*)pConnInfo->remoteAddress)->sin6_addr, sAddr, sizeof(sAddr)))
			{
				sRemoteAddr = sAddr;
			}

			printf("local=[%s]:%d remote=[%s]:%d ", 
							sLocalAddr.c_str(), ntohs(((sockaddr_in6*)pConnInfo->localAddress)->sin6_port),
							sRemoteAddr.c_str(), ntohs(((sockaddr_in6*)pConnInfo->remoteAddress)->sin6_port));
		}
	
		unsigned int uid = 0;
		if (nf_getUid(id, &uid) == NF_STATUS_SUCCESS)
		{
			if (nf_getUserName(uid, path, sizeof(path))  == NF_STATUS_SUCCESS)
	    		printf("user %u: %s\n", uid, path);
		}

		if (nf_getProcessName(pConnInfo->processId, path, sizeof(path)) == NF_STATUS_SUCCESS)
		{
			printf("process %lu: %s\n", pConnInfo->processId, path);
		} else
		{
			printf("PID %lu:\n", pConnInfo->processId);
		} 

		fflush(stdout);
	}

	virtual void tcpClosed(ENDPOINT_ID id, PNF_TCP_CONN_INFO pConnInfo)
	{
		printf("tcpClosed id=%llu, conn.count=%lu\n", id, nf_getConnCount());
		fflush(stdout);
	}

	virtual void tcpReceive(ENDPOINT_ID id, const char * buf, int len)
	{	
		printf("tcpReceive id=%llu len=%u\n", id, len);
		fflush(stdout);

		// Send the packet to application
		nf_tcpPostReceive(id, buf, len);
	}

	virtual void tcpSend(ENDPOINT_ID id, const char * buf, int len)
	{
		printf("tcpSend id=%llu len=%u\n", id, len);
		fflush(stdout);

		// Send the packet to server
		nf_tcpPostSend(id, buf, len);
	}

	virtual void tcpCanReceive(ENDPOINT_ID id)
	{
		printf("tcpCanReceive id=%llu\n", id);
		fflush(stdout);
	}

	virtual void tcpCanSend(ENDPOINT_ID id)
	{
		printf("tcpCanSend id=%llu\n", id);
		fflush(stdout);
	}

	//
	// UDP events
	//

	virtual void udpCreated(ENDPOINT_ID id, PNF_UDP_CONN_INFO pConnInfo)
	{
	}

	virtual void udpConnectRequest(ENDPOINT_ID id, PNF_UDP_CONN_REQUEST pConnReq)
	{
	}

	virtual void udpClosed(ENDPOINT_ID id, PNF_UDP_CONN_INFO pConnInfo)
	{
	}

	virtual void udpReceive(ENDPOINT_ID id, const unsigned char * remoteAddress, const char * buf, int len, PNF_UDP_OPTIONS options)
	{	
	}

	virtual void udpSend(ENDPOINT_ID id, const unsigned char * remoteAddress, const char * buf, int len, PNF_UDP_OPTIONS options)
	{
	}

	virtual void udpCanReceive(ENDPOINT_ID id)
	{
	}

	virtual void udpCanSend(ENDPOINT_ID id)
	{
	}
};

int main(int argc, char* argv[])
{
	EventHandler eh;
	NFEXT_RULE rule;

	printf("Start\n");

	if (nf_requireFileLimit(12000) != NF_STATUS_SUCCESS)
	{
		printf("Unable to raise the limit for number of files\n");
		return -1;
	}

	printf("Press enter to stop...\n\n");

	nf_setOptions(0, 0);
   
	// Initialize the library and start filtering thread
	if (nf_init(NFDRIVER_NAME, &eh) != NF_STATUS_SUCCESS)
	{
		printf("Failed to connect to driver");
		return -1;
	}

	NFEXT_RULE rules[4];
	PNFEXT_RULE pRule;
	
	pRule = &rules[0];

	memset(pRule, 0, sizeof(NFEXT_RULE));
	pRule->ruleType = NFEXT_PACKET_RULE;
	pRule->protocol = IPPROTO_UDP;
	pRule->fieldsMask = NFEXT_USE_DST_PORTS;
	pRule->dstPorts.from = pRule->dstPorts.to = 443;
	pRule->filteringFlag = NFEXT_BLOCK;

	pRule = &rules[1];

	memset(pRule, 0, sizeof(NFEXT_RULE));
	pRule->ruleType = NFEXT_PACKET_RULE;
	pRule->protocol = IPPROTO_UDP;
	pRule->fieldsMask = NFEXT_USE_DST_PORTS;
	pRule->dstPorts.from = pRule->dstPorts.to = 80;
	pRule->filteringFlag = NFEXT_BLOCK;

	pRule = &rules[2];

	memset(pRule, 0, sizeof(NFEXT_RULE));
	pRule->ruleType = NFEXT_REDIRECT_RULE;
	pRule->protocol = IPPROTO_TCP;
	pRule->direction = NF_D_IN;
	pRule->fieldsMask = NFEXT_USE_DST_PORTS;
	pRule->dstPorts.from = pRule->dstPorts.to = 22;
	pRule->filteringFlag = NFEXT_BYPASS;

	pRule = &rules[3];

	memset(pRule, 0, sizeof(NFEXT_RULE));
	pRule->ruleType = NFEXT_REDIRECT_RULE;
	pRule->filteringFlag = NFEXT_REDIRECT;

	if (nf_setRules(rules, 4) != NF_STATUS_SUCCESS)
	{
		printf("Failed to add filtering rules. Root rights are required.\n");
		return -1;
	}

	// Wait for Enter
	getchar();

	nf_deleteRules();

	// Free the library
	nf_free();

	printf("Stopped\n");

	return 0;
}

