/**
	This sample allows to redirect TCP connections with given parameters to the specified endpoint.
	It also supports redirecting to HTTPS or SOCKS proxy.

	Usage: TcpRedirector.exe [-s|t] [-p Port] [-process ProcessName] -r IP:Port 
		-t : redirect via HTTPS proxy at IP:Port. The proxy must support HTTP tunneling (CONNECT requests)
		-s : redirect via SOCKS4 proxy at IP:Port. 
		Port : remote port to intercept
		ProcessName : redirect connections of the process with specified substring in path
		IP:Port : redirect connections to the specified IP endpoint

	Example:
		
		TcpRedirector.exe -t -p 110 -r 163.15.64.8:3128

		The sample will redirect POP3 connections to HTTPS proxy at 163.15.64.8:3128 with tunneling to real server. 
**/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <string>
#include <map>
#include <vector>

#include "nfapi_linux.h"

using namespace nfapi;

#define NFEXT_BUNDLEID	"nfext"
#define NFDRIVER_NAME NFEXT_BUNDLEID

#pragma pack(push, 1)

enum eSOCKS_VERSION
{
	SOCKS_4 = 4,
	SOCKS_5 = 5
};

enum eSOCKS4_COMMAND
{
	S4C_CONNECT = 1,
	S4C_BIND = 2
};

struct SOCKS4_REQUEST
{
	char	version;
	char	command;
	unsigned short port;
	unsigned int ip;
	char userid[1];
};

enum eSOCKS4_RESCODE
{
	S4RC_SUCCESS = 0x5a
};

struct SOCKS4_RESPONSE
{
	char	reserved;
	char	res_code;
	unsigned short reserved1;
	unsigned int reserved2;
};

#pragma pack(pop)

// Forward declarations
void printConnInfo(bool connected, ENDPOINT_ID id, PNF_TCP_CONN_INFO pConnInfo);

unsigned char	g_redirectToAddress[NF_MAX_ADDRESS_LENGTH];
std::string 	g_processName;

enum PROXY_TYPE
{
	PT_NONE,
	PT_HTTPS,
	PT_SOCKS
};

PROXY_TYPE g_proxyType = PT_NONE;

bool stringToAddress(const char * sAddr, char * addr, int addrLen)
{
	char * p;
	unsigned short port = 0;
	sockaddr_in * addrIn = (sockaddr_in *)addr;
	sockaddr_in6 * addrIn6 = (sockaddr_in6 *)addr;
	char sIPAddr[100];
	
	p = strchr((char*)sAddr, ':');
	if (!p)
	{
		return false;
	}

	memcpy(sIPAddr, sAddr, p - sAddr);
	sIPAddr[p-sAddr] = '\0';

	port = atoi(p+1);
	if (port == 0)
	{
		return false;	
	}

	memset(addr, 0, addrLen);

	if (inet_pton(AF_INET, sIPAddr, &addrIn->sin_addr) > 0)
	{
		addrIn->sin_family = AF_INET;
		addrIn->sin_port = htons(port);
		return true;
	} else
	if (inet_pton(AF_INET6, sIPAddr, &addrIn6->sin6_addr) > 0)
	{
		addrIn6->sin6_family = AF_INET6;
		addrIn6->sin6_port = htons(port);
		return true;
	}

	return false;
}

bool addressToString(char * addr, std::string addrString)
{
	char sAddr[120];
	sockaddr * pAddr;

	pAddr = (sockaddr*)addr;

	if (pAddr->sa_family == AF_INET)
	{
		if (inet_ntop(AF_INET, &((sockaddr_in*)addr)->sin_addr, sAddr, sizeof(sAddr)) == NULL)
			return false;
		addrString = sAddr;
		snprintf(sAddr, sizeof(sAddr), ":%d", ntohs(((sockaddr_in*)addr)->sin_port));
		addrString += sAddr;
	} else
	if (pAddr->sa_family == AF_INET6)
	{
		if (inet_ntop(AF_INET6, &((sockaddr_in6*)addr)->sin6_addr, sAddr, sizeof(sAddr)) == NULL)
			return false;
		addrString = sAddr;
		snprintf(sAddr, sizeof(sAddr), ":%d", ntohs(((sockaddr_in6*)addr)->sin6_port));
		addrString += sAddr;
	} else
		return false;

	return true;	
}

//
//	API events handler
//
class EventHandler : public NF_EventHandler
{
	struct ORIGINAL_CONN_INFO
	{
		unsigned char	remoteAddress[NF_MAX_ADDRESS_LENGTH];
		std::vector<char>	pendedSends;
	};

	typedef std::map<ENDPOINT_ID, ORIGINAL_CONN_INFO> tConnInfoMap;
	tConnInfoMap m_connInfoMap;

	virtual void threadStart()
	{
		printf("threadStart\n");
		fflush(stdout);

		// Initialize thread specific stuff
	}

	virtual void threadEnd()
	{
		printf("threadEnd\n");

		// Uninitialize thread specific stuff
	}
	
	//
	// TCP events
	//

	virtual void tcpConnectRequest(ENDPOINT_ID id, PNF_TCP_CONN_INFO pConnInfo)
	{
		char path[MAX_PATH];
		sockaddr * pAddr = (sockaddr*)pConnInfo->remoteAddress;

		printf("tcpConnectRequest id=%llu\n", id);

		if (pAddr->sa_family == AF_INET6)
			return;

		if (nf_getProcessName(pConnInfo->processId, path, sizeof(path)) == NF_STATUS_SUCCESS)
		{
			printf("process %lu: %s\n", pConnInfo->processId, path);
		 	if (!g_processName.empty())
			{
				if (strcasestr(path, g_processName.c_str()) == NULL)
				{
					nf_tcpDisableFiltering(id);
					return;
				}
			}
		}

		if (g_proxyType != PT_NONE)
		{
			ORIGINAL_CONN_INFO oci;
			memcpy(oci.remoteAddress, pConnInfo->remoteAddress, sizeof(oci.remoteAddress));

			// Save the original remote address
			m_connInfoMap[id] = oci;
		}

		// Redirect the connection if it is not already redirected
		if (memcmp(pAddr, g_redirectToAddress, sizeof(sockaddr_in)) != 0)
		{
			// Change the remote address
			memcpy(pConnInfo->remoteAddress, g_redirectToAddress, sizeof(pConnInfo->remoteAddress));
		} 
	}

	virtual void tcpConnected(ENDPOINT_ID id, PNF_TCP_CONN_INFO pConnInfo)
	{
		printf("tcpConnected id=%llu\n", id);
		fflush(stdout);
	}

	virtual void tcpClosed(ENDPOINT_ID id, PNF_TCP_CONN_INFO pConnInfo)
	{
		printf("tcpClosed id=%llu, conn.count=%lu\n", id, nf_getConnCount());
		fflush(stdout);

		if (g_proxyType != PT_NONE)
		{
			m_connInfoMap.erase(id);
		}
	}

	virtual void tcpReceive(ENDPOINT_ID id, const char * buf, int len)
	{	
		printf("tcpReceive id=%llu len=%u\n", id, len);
		fflush(stdout);

		if (g_proxyType != PT_NONE)
		{
			tConnInfoMap::iterator it;

			it = m_connInfoMap.find(id);
			if (it != m_connInfoMap.end())
			{
				if (!it->second.pendedSends.empty())
				{
					nf_tcpPostSend(id, &it->second.pendedSends[0], (int)it->second.pendedSends.size());
				}

				m_connInfoMap.erase(id);
				// The first packet is a response from proxy server.
				// Skip it.

				return;
			}
		}

		// Send the packet to application
		nf_tcpPostReceive(id, buf, len);

		// Don't filter the subsequent packets (optimization)
		nf_tcpDisableFiltering(id);
	}

	virtual void tcpSend(ENDPOINT_ID id, const char * buf, int len)
	{
		printf("tcpSend id=%llu len=%u\n", id, len);
		fflush(stdout);

		if (g_proxyType != PT_NONE)
		{
			tConnInfoMap::iterator it;

			it = m_connInfoMap.find(id);
			
			if (it != m_connInfoMap.end())
			{
				switch (g_proxyType)
				{
				case PT_NONE:
					nf_tcpDisableFiltering(id);
					break;

				case PT_HTTPS:
					{
						char request[200];
						std::string sAddr;
						tConnInfoMap::iterator it;

						// Generate CONNECT request using saved original remote address

						it = m_connInfoMap.find(id);
						if (it == m_connInfoMap.end())
							return;
						
						if (!addressToString((char*)it->second.remoteAddress, sAddr))
						{
							return;
						}

						_snprintf(request, sizeof(request), "CONNECT %s HTTP/1.0\r\n\r\n", sAddr.c_str());

						// Send the request first
						nf_tcpPostSend(id, request, (int)strlen(request));

						it->second.pendedSends.insert(it->second.pendedSends.end(), buf, buf+len);
						return;
					}
					break;

				case PT_SOCKS:
					{
						SOCKS4_REQUEST request;
						sockaddr_in * pAddr;
						tConnInfoMap::iterator it;

						// Generate SOCKS request using saved original remote address

						it = m_connInfoMap.find(id);
						if (it == m_connInfoMap.end())
							return;
						
						pAddr = (sockaddr_in*)&it->second.remoteAddress;
						if (pAddr->sin_family == AF_INET6)
						{
							return;
						}

						request.version = SOCKS_4;
						request.command = S4C_CONNECT;
						request.ip = pAddr->sin_addr.s_addr;
						request.port = pAddr->sin_port;
						request.userid[0] = '\0';

						// Send the request first
						nf_tcpPostSend(id, (char*)&request, (int)sizeof(request));

						it->second.pendedSends.insert(it->second.pendedSends.end(), buf, buf+len);

						return;
					}
					break;
				}
			}
		}

		// Send the packet to server
		nf_tcpPostSend(id, buf, len);
	}

	virtual void tcpCanReceive(ENDPOINT_ID id)
	{
	}

	virtual void tcpCanSend(ENDPOINT_ID id)
	{
	}
	
	//
	// UDP events
	//

	virtual void udpCreated(ENDPOINT_ID id, PNF_UDP_CONN_INFO pConnInfo)
	{
	}

	virtual void udpConnectRequest(ENDPOINT_ID id, PNF_UDP_CONN_REQUEST pConnReq)
	{
	}

	virtual void udpClosed(ENDPOINT_ID id, PNF_UDP_CONN_INFO pConnInfo)
	{
	}

	virtual void udpReceive(ENDPOINT_ID id, const unsigned char * remoteAddress, const char * buf, int len, PNF_UDP_OPTIONS options)
	{	
	}

	virtual void udpSend(ENDPOINT_ID id, const unsigned char * remoteAddress, const char * buf, int len, PNF_UDP_OPTIONS options)
	{
	}

	virtual void udpCanReceive(ENDPOINT_ID id)
	{
	}

	virtual void udpCanSend(ENDPOINT_ID id)
	{
	}
};


void usage()
{
	printf("Usage: TcpRedirector.exe [-t|s] [-p Port] [-process ProcessName] -r IP:Port\n" \
		"Port : remote port to intercept\n" \
		"ProcessName : redirect connections of the process with specified substring in path\n" \
		"IP:Port : redirect connections to the specified IP endpoint\n" \
		"-t : turn on tunneling via proxy at IP:Port. The proxy must support HTTP tunneling (CONNECT requests)\n" \
		"-s : redirect via anonymous SOCKS4 proxy at IP:Port.\n "
		);
	exit(0);
}

int main(int argc, char* argv[])
{
	EventHandler eh;
	NFEXT_RULE rule;
	bool redirectAddressSpecified = false;

	if (nf_requireFileLimit(12000) != NF_STATUS_SUCCESS)
	{
		printf("Unable to raise the limit for number of files\n");
		return -1;
	}

	nf_setOptions(0, 0);

	memset(&rule, 0, sizeof(rule));
	rule.ruleType = NFEXT_REDIRECT_RULE;
	rule.direction = NF_D_OUT;
	rule.filteringFlag = NFEXT_REDIRECT;

	for (int i=1; i < argc; i += 2)
	{
		if (strcmp(argv[i], "-p") == 0)
		{
			rule.fieldsMask |= NFEXT_USE_DST_PORTS;
			rule.dstPorts.from = rule.dstPorts.to = atoi(argv[i+1]);
			printf("Remote port: %s\n", argv[i+1]);
		} else
		if (strcmp(argv[i], "-process") == 0)
		{
			g_processName = argv[i+1];	
			printf("Process name: %s\n", argv[i+1]);
		} else
		if (strcmp(argv[i], "-t") == 0)
		{
			g_proxyType = PT_HTTPS;	
			i--;
			printf("Tunnel via HTTP proxy\n");
		} else
		if (strcmp(argv[i], "-s") == 0)
		{
			g_proxyType = PT_SOCKS;	
			i--;
			printf("Tunnel via SOCKS proxy\n");
		} else
		if (strcmp(argv[i], "-r") == 0)
		{
			char * p;
			
			memset(&g_redirectToAddress, 0, sizeof(g_redirectToAddress));
	
			if (!stringToAddress(argv[i+1], (char*)g_redirectToAddress, sizeof(g_redirectToAddress)))
			{
				printf("invalid address %s\n", argv[i+1]);
				usage();
		    }

			redirectAddressSpecified = true;

			printf("Redirecting to: %s\n", argv[i+1]);
		} else
		{
			usage();
		}
	}

	if (!redirectAddressSpecified)
	{
		usage();
	}

	// Initialize the library and start filtering thread
	if (nf_init(NFDRIVER_NAME, &eh) != NF_STATUS_SUCCESS)
	{
		printf("Failed to connect to driver");
		return -1;
	}

	nf_setRules(&rule, 1);

	printf("Press Enter to stop...\n\n");

	// Wait for any key
	getchar();

	// Free the library
	nf_free();

	return 0;
}

